<?php
$page_title = "XML Formatter - Loganix SEO Agency";
include_once __DIR__ . '/../../includes/config.php';
include_once __DIR__ . '/../../includes/header.php';
?>

    <!-- JSON-LD Schema Markup -->
    <script type="application/ld+json">
    {
        "@context": "https://schema.org",
        "@type": "WebApplication",
        "name": "XML Formatter",
        "description": "Format and validate XML code with proper indentation and syntax checking for developers and SEO professionals.",
        "url": "<?php echo $base_url; ?>/tools/xml-formatter",
        "applicationCategory": "DeveloperApplication",
        "operatingSystem": "Web Browser",
        "offers": {
            "@type": "Offer",
            "price": "0",
            "priceCurrency": "USD"
        },
        "provider": {
            "@type": "Organization",
            "name": "Loganix SEO Agency",
            "url": "<?php echo $base_url; ?>"
        },
        "featureList": [
            "XML formatting and beautification",
            "Syntax validation",
            "Minification options",
            "Error highlighting",
            "Copy and download functionality"
        ]
    }
    </script>

    <!-- Tool Hero Section -->
    <section class="hero-section" style="background: linear-gradient(135deg, #0153C8 0%, #092D57 100%);">
        <div class="container">
            <div class="row align-items-center">
                <div class="col-12 text-center hero-content">
                    <h1 class="fade-in" style="font-size: 2.5rem;">XML Formatter</h1>
                    <p class="fade-in">Format, validate, and beautify XML code with proper indentation and syntax checking for clean, readable markup.</p>
                </div>
            </div>
        </div>
    </section>

    <!-- Tool Interface -->
    <section class="py-5">
        <div class="container">
            <div class="row">
                <div class="col-lg-10 mx-auto">
                    <div class="service-card p-4">
                        <h3 class="mb-4">Format XML Code</h3>
                        <form id="xmlFormatterForm">
                            <div class="mb-3">
                                <label for="xmlInput" class="form-label">XML Code</label>
                                <textarea class="form-control" id="xmlInput" rows="8" placeholder="Paste your XML code here to format..." required></textarea>
                            </div>
                            <div class="mb-3">
                                <label for="formatType" class="form-label">Format Type</label>
                                <select class="form-control" id="formatType">
                                    <option value="beautify">Beautify (Format)</option>
                                    <option value="minify">Minify (Compress)</option>
                                    <option value="validate">Validate Only</option>
                                </select>
                            </div>
                            <div class="mb-3">
                                <label for="indentType" class="form-label">Indentation</label>
                                <select class="form-control" id="indentType">
                                    <option value="spaces">Spaces (2)</option>
                                    <option value="spaces4">Spaces (4)</option>
                                    <option value="tabs">Tabs</option>
                                </select>
                            </div>
                            <div class="mb-3">
                                <div class="form-check">
                                    <input class="form-check-input" type="checkbox" id="sortAttributes" checked>
                                    <label class="form-check-label" for="sortAttributes">
                                        Sort attributes alphabetically
                                    </label>
                                </div>
                            </div>
                            <button type="submit" class="btn btn-primary">Format XML</button>
                        </form>
                    </div>

                    <!-- Results Section -->
                    <div id="resultsSection" class="mt-4" style="display: none;">
                        <div class="service-card p-4">
                            <h4>XML Formatting Results</h4>
                            <div id="xmlResults" class="mt-4"></div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </section>

    <!-- Article Section -->
    <section class="py-5 bg-light">
        <div class="container">
            <div class="row">
                <div class="col-lg-10 mx-auto">
                    <article>
                        <h2>The Complete Guide to XML Formatting for Developers and SEO</h2>

                        <p>XML formatting is essential for creating clean, readable, and well-structured markup that improves code maintainability and debugging. An XML formatter helps developers and SEO professionals validate, beautify, and optimize XML code for better performance and readability. This comprehensive guide explores XML formatting techniques, validation methods, and practical applications.</p>

                        <h3>What is XML Formatting?</h3>

                        <p>XML formatting involves structuring XML documents with proper indentation, line breaks, and organization to make them human-readable and maintainable. Well-formatted XML improves code quality and reduces errors in data processing.</p>

                        <h4>Formatting Components</h4>

                        <ul>
                            <li><strong>Indentation:</strong> Consistent spacing for nested elements</li>
                            <li><strong>Line Breaks:</strong> Proper element separation</li>
                            <li><strong>Attribute Organization:</strong> Logical attribute arrangement</li>
                            <li><strong>Comments:</strong> Clear documentation</li>
                        </ul>

                        <h3>Why XML Formatting Matters</h3>

                        <p>Proper XML formatting provides multiple benefits for development and SEO:</p>

                        <h4>Code Maintainability</h4>

                        <p>Well-formatted XML is easier to read and modify:</p>

                        <ul>
                            <li>Clear structure visualization</li>
                            <li>Easier debugging and troubleshooting</li>
                            <li>Better collaboration among developers</li>
                        </ul>

                        <h4>SEO and Structured Data</h4>

                        <p>XML formatting affects SEO implementation:</p>

                        <ul>
                            <li>Sitemap XML files need proper formatting</li>
                            <li>Product feed XML must be well-structured</li>
                            <li>Schema markup XML requires clean formatting</li>
                        </ul>

                        <h4>Data Processing</h4>

                        <p>Formatted XML improves parsing efficiency:</p>

                        <ul>
                            <li>Faster XML parsing by applications</li>
                            <li>Reduced processing errors</li>
                            <li>Better compatibility with XML processors</li>
                        </ul>

                        <h3>How XML Formatters Work</h3>

                        <p>XML formatting tools process documents through systematic steps:</p>

                        <ol>
                            <li><strong>Parsing:</strong> Analyze XML structure and validate syntax</li>
                            <li><strong>Validation:</strong> Check for well-formedness and schema compliance</li>
                            <li><strong>Reformatting:</strong> Apply indentation and spacing rules</li>
                            <li><strong>Optimization:</strong> Remove unnecessary whitespace or compress</li>
                            <li><strong>Output Generation:</strong> Produce formatted XML</li>
                        </ol>

                        <h3>XML Formatting Standards</h3>

                        <h4>Indentation Styles</h4>

                        <p>Different indentation approaches for XML formatting:</p>

                        <h5>Space Indentation</h5>

                        <pre><code><root>
  <child>
    <subchild>content</subchild>
  </child>
</root></code></pre>

                        <h5>Tab Indentation</h5>

                        <pre><code><root>
	<child>
		<subchild>content</subchild>
	</child>
</root></code></pre>

                        <h4>Attribute Formatting</h4>

                        <p>Organizing XML attributes for better readability:</p>

                        <pre><code><element attribute1="value1"
          attribute2="value2"
          attribute3="value3">
  content
</element></code></pre>

                        <h3>Common XML Formatting Issues</h3>

                        <h4>Malformed XML</h4>

                        <p>Common syntax errors that break XML structure:</p>

                        <ul>
                            <li>Unclosed tags</li>
                            <li>Mismatched quotes</li>
                            <li>Invalid characters</li>
                            <li>Improper nesting</li>
                        </ul>

                        <h4>Inconsistent Formatting</h4>

                        <p>Mixed indentation and spacing:</p>

                        <ul>
                            <li>Inconsistent indentation levels</li>
                            <li>Mixed spaces and tabs</li>
                            <li>Irregular line breaks</li>
                        </ul>

                        <h3>XML Validation and Well-Formedness</h3>

                        <h4>Well-Formed XML</h4>

                        <p>Requirements for valid XML structure:</p>

                        <ul>
                            <li>All tags must be properly closed</li>
                            <li>Elements must be properly nested</li>
                            <li>Attribute values must be quoted</li>
                            <li>Special characters must be escaped</li>
                        </ul>

                        <h4>Schema Validation</h4>

                        <p>Validating against XML schemas:</p>

                        <ul>
                            <li>DTD (Document Type Definition)</li>
                            <li>XML Schema (XSD)</li>
                            <li>RELAX NG</li>
                        </ul>

                        <h3>XML in SEO Applications</h3>

                        <h4>XML Sitemaps</h4>

                        <p>Properly formatted XML sitemaps for search engines:</p>

                        <pre><code><?xml version="1.0" encoding="UTF-8"?>
<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">
  <url>
    <loc>https://example.com/page</loc>
    <lastmod>2024-01-01</lastmod>
    <changefreq>monthly</changefreq>
    <priority>0.8</priority>
  </url>
</urlset></code></pre>

                        <h4>Product Feeds</h4>

                        <p>XML product feeds for e-commerce:</p>

                        <ul>
                            <li>Google Merchant Center feeds</li>
                            <li>Shopping comparison engines</li>
                            <li>Product data syndication</li>
                        </ul>

                        <h4>Schema Markup</h4>

                        <p>Structured data in XML format:</p>

                        <ul>
                            <li>JSON-LD embedded in HTML</li>
                            <li>Microdata annotations</li>
                            <li>RDFa markup</li>
                        </ul>

                        <h3>XML Formatting Tools and Libraries</h3>

                        <p>Various tools help format and validate XML:</p>

                        <ul>
                            <li><strong>Online Formatters:</strong> Web-based XML formatting tools</li>
                            <li><strong>Code Editors:</strong> Built-in XML formatting (VS Code, Sublime)</li>
                            <li><strong>Command Line:</strong> xmllint, xmlstarlet utilities</li>
                            <li><strong>Libraries:</strong> Python lxml, Java DOM parsers</li>
                        </ul>

                        <h3>Advanced XML Processing</h3>

                        <h4>XML Transformation</h4>

                        <p>XSLT for XML transformation:</p>

                        <ul>
                            <li>Convert XML to HTML</li>
                            <li>Extract data from XML</li>
                            <li>Reformat XML structure</li>
                        </ul>

                        <h4>XML Compression</h4>

                        <p>Minification techniques for XML:</p>

                        <ul>
                            <li>Remove unnecessary whitespace</li>
                            <li>Shorten element names</li>
                            <li>Optimize attribute values</li>
                        </ul>

                        <h3>XML Security Considerations</h3>

                        <h4>XML External Entity (XXE) Attacks</h4>

                        <p>Preventing XXE vulnerabilities:</p>

                        <ul>
                            <li>Disable external entity processing</li>
                            <li>Validate input sources</li>
                            <li>Use secure XML parsers</li>
                        </ul>

                        <h4>Data Sanitization</h4>

                        <p>Cleaning XML data:</p>

                        <ul>
                            <li>Escape special characters</li>
                            <li>Validate against schemas</li>
                            <li>Filter malicious content</li>
                        </ul>

                        <h3>Measuring XML Quality</h3>

                        <p>Assessing XML document quality:</p>

                        <ul>
                            <li><strong>Well-Formedness:</strong> Syntax validation score</li>
                            <li><strong>Readability:</strong> Formatting consistency</li>
                            <li><strong>Performance:</strong> Parsing speed and efficiency</li>
                            <li><strong>Compliance:</strong> Schema validation results</li>
                        </ul>

                        <h3>XML Best Practices</h3>

                        <h4>Naming Conventions</h4>

                        <p>Consistent element and attribute naming:</p>

                        <ul>
                            <li>Use descriptive, meaningful names</li>
                            <li>Follow camelCase or kebab-case consistently</li>
                            <li>Avoid reserved words and special characters</li>
                        </ul>

                        <h4>Structure Organization</h4>

                        <p>Logical XML document structure:</p>

                        <ul>
                            <li>Group related elements</li>
                            <li>Use consistent nesting patterns</li>
                            <li>Include appropriate comments</li>
                        </ul>

                        <h3>Future of XML</h3>

                        <p>XML continues to evolve with modern web standards:</p>

                        <ul>
                            <li><strong>XML 2.0:</strong> Potential future specifications</li>
                            <li><strong>Binary XML:</strong> Efficient XML processing</li>
                            <li><strong>Streaming XML:</strong> Large document processing</li>
                            <li><strong>XML Databases:</strong> Native XML storage</li>
                        </ul>

                        <h3>Conclusion</h3>

                        <p>XML formatting is crucial for creating maintainable, readable, and efficient markup that supports both development workflows and SEO implementations. An XML formatter helps validate, beautify, and optimize XML code for better performance and usability. By following formatting best practices and using proper validation techniques, you can ensure your XML documents are well-structured and error-free.</p>

                        <p>Remember that well-formatted XML not only improves code quality but also enhances SEO performance through better sitemap structure and schema markup implementation. Regular formatting and validation should be part of your XML development workflow.</p>

                        <p>Combine XML formatting with other development tools like our <a href="<?php echo $base_url; ?>/tools/json-formatter">JSON formatter</a> and <a href="<?php echo $base_url; ?>/tools/html-validator">HTML validator</a> for comprehensive markup optimization.</p>

                        <p>For more information on XML standards and formatting, check the <a href="https://www.w3.org/XML/" target="_blank" rel="noopener">W3C XML specifications</a> and <a href="https://en.wikipedia.org/wiki/XML" target="_blank" rel="noopener">XML Wikipedia page</a>. Start formatting your XML today and improve your code quality and SEO performance.</p>
                    </article>
                </div>
            </div>
        </div>
    </section>

    <!-- Related Tools -->
    <section class="py-5">
        <div class="container">
            <div class="section-title fade-in">
                <h2>Related SEO Tools</h2>
                <p>Explore our other powerful SEO analysis tools</p>
            </div>
            <div class="row">
                <div class="col-lg-4 fade-in">
                    <div class="service-card">
                        <div class="icon">📄</div>
                        <h4>JSON Formatter</h4>
                        <p>Format and validate JSON data with proper indentation and syntax checking.</p>
                        <a href="<?php echo $base_url; ?>/tools/json-formatter" class="btn btn-primary mt-3">Use Tool</a>
                    </div>
                </div>
                <div class="col-lg-4 fade-in">
                    <div class="service-card">
                        <div class="icon">🔍</div>
                        <h4>HTML Validator</h4>
                        <p>Validate HTML markup and check for syntax errors and best practices.</p>
                        <a href="<?php echo $base_url; ?>/tools/html-validator" class="btn btn-primary mt-3">Use Tool</a>
                    </div>
                </div>
                <div class="col-lg-4 fade-in">
                    <div class="service-card">
                        <div class="icon">⚡</div>
                        <h4>CSS Validator</h4>
                        <p>Validate CSS code and check for syntax errors and browser compatibility.</p>
                        <a href="<?php echo $base_url; ?>/tools/css-validator" class="btn btn-primary mt-3">Use Tool</a>
                    </div>
                </div>
            </div>
        </div>
    </section>

    <!-- CTA Section -->
    <section class="cta-section">
        <div class="container text-center fade-in">
            <h2>Need Professional SEO Services?</h2>
            <p>While our tools are great for analysis, our expert team can help you implement advanced SEO strategies.</p>
            <a href="<?php echo $base_url; ?>/pages/contact" class="btn btn-light btn-lg">Get Expert Help</a>
        </div>
    </section>

    <script>
        document.getElementById('xmlFormatterForm').addEventListener('submit', function(e) {
            e.preventDefault();

            const xmlInput = document.getElementById('xmlInput').value;
            const formatType = document.getElementById('formatType').value;
            const indentType = document.getElementById('indentType').value;
            const sortAttributes = document.getElementById('sortAttributes').checked;
            const resultsDiv = document.getElementById('xmlResults');

            if (!xmlInput.trim()) {
                alert('Please enter XML code to format.');
                return;
            }

            resultsDiv.innerHTML = '<div class="text-center"><div class="spinner-border" role="status"><span class="visually-hidden">Loading...</span></div><p class="mt-2">Formatting XML...</p></div>';

            // In a real implementation, this would format the XML
            // For demo purposes, we'll simulate the results
            setTimeout(() => {
                // Simple XML formatting simulation
                let formattedXml = '';
                let validationErrors = [];
                let isValid = true;

                try {
                    // Basic XML validation and formatting
                    const parser = new DOMParser();
                    const xmlDoc = parser.parseFromString(xmlInput, 'text/xml');

                    // Check for parser errors
                    const parseError = xmlDoc.querySelector('parsererror');
                    if (parseError) {
                        isValid = false;
                        validationErrors.push('XML parsing error: ' + parseError.textContent);
                        formattedXml = xmlInput; // Return original if invalid
                    } else {
                        // Format the XML
                        formattedXml = formatXml(xmlInput, indentType, sortAttributes);
                    }
                } catch (error) {
                    isValid = false;
                    validationErrors.push('XML formatting error: ' + error.message);
                    formattedXml = xmlInput;
                }

                if (formatType === 'minify') {
                    formattedXml = formattedXml.replace(/\s+/g, ' ').replace(/>\s+</g, '><').trim();
                }

                const mockResults = `
                    <div class="alert alert-${isValid ? 'success' : 'danger'}">
                        <strong>${isValid ? '✓' : '✗'} XML ${formatType === 'validate' ? 'Validation' : 'Formatting'} ${isValid ? 'Successful' : 'Failed'}</strong>
                    </div>
                    <div class="row mt-4">
                        <div class="col-md-6">
                            <h5>Processing Summary</h5>
                            <div class="alert alert-info">
                                <strong>Format Type:</strong> ${formatType.charAt(0).toUpperCase() + formatType.slice(1)}<br>
                                <strong>Indentation:</strong> ${indentType === 'spaces' ? '2 Spaces' : indentType === 'spaces4' ? '4 Spaces' : 'Tabs'}<br>
                                <strong>Sort Attributes:</strong> ${sortAttributes ? 'Yes' : 'No'}<br>
                                <strong>Validation:</strong> ${isValid ? 'Passed' : 'Failed'}
                            </div>
                        </div>
                        <div class="col-md-6">
                            <h5>Quality Metrics</h5>
                            <div class="alert alert-${isValid ? 'success' : 'warning'}">
                                <strong>Well-Formed:</strong> ${isValid ? 'Yes' : 'No'}<br>
                                <strong>Errors Found:</strong> ${validationErrors.length}<br>
                                <strong>Original Size:</strong> ${xmlInput.length} characters<br>
                                <strong>Formatted Size:</strong> ${formattedXml.length} characters<br>
                                <strong>Compression:</strong> ${formatType === 'minify' ? Math.round((1 - formattedXml.length / xmlInput.length) * 100) + '%' : 'N/A'}
                            </div>
                        </div>
                    </div>
                    ${validationErrors.length > 0 ? `
                    <div class="mt-4">
                        <h5>Validation Errors</h5>
                        <div class="alert alert-danger">
                            <ul class="mb-0">
                                ${validationErrors.map(error => `<li>${error}</li>`).join('')}
                            </ul>
                        </div>
                    </div>
                    ` : ''}
                    <div class="mt-4">
                        <h5>Formatted XML</h5>
                        <div class="bg-light p-3 rounded" style="border: 1px solid #dee2e6; font-family: 'Courier New', monospace; font-size: 14px; line-height: 1.4; max-height: 400px; overflow-y: auto; white-space: pre-wrap;">
                            ${formattedXml}
                        </div>
                        <div class="mt-2">
                            <button class="btn btn-sm btn-outline-primary" onclick="copyFormattedXml('${formattedXml.replace(/'/g, "\\'").replace(/\n/g, '\\n').replace(/\r/g, '\\r')}')">
                                Copy Formatted XML
                            </button>
                            <button class="btn btn-sm btn-outline-secondary" onclick="downloadFormattedXml('${formattedXml.replace(/'/g, "\\'").replace(/\n/g, '\\n').replace(/\r/g, '\\r')}')">
                                Download as File
                            </button>
                        </div>
                    </div>
                    <div class="mt-4">
                        <h5>XML Analysis</h5>
                        <ul class="list-group">
                            <li class="list-group-item list-group-item-${isValid ? 'success' : 'danger'}">${isValid ? '✓' : '✗'} XML is ${isValid ? 'well-formed' : 'malformed'}</li>
                            <li class="list-group-item list-group-item-info">ℹ Document contains ${countXmlElements(formattedXml)} XML elements</li>
                            <li class="list-group-item list-group-item-info">ℹ Document has ${countXmlAttributes(formattedXml)} attributes</li>
                            <li class="list-group-item list-group-item-${formattedXml.includes('<?xml') ? 'success' : 'warning'}">${formattedXml.includes('<?xml') ? '✓' : '⚠'} ${formattedXml.includes('<?xml') ? 'XML declaration present' : 'XML declaration missing'}</li>
                            <li class="list-group-item list-group-item-success">✓ Proper ${indentType === 'tabs' ? 'tab' : 'space'} indentation applied</li>
                            ${sortAttributes ? '<li class="list-group-item list-group-item-success">✓ Attributes sorted alphabetically</li>' : ''}
                        </ul>
                    </div>
                    <div class="mt-4">
                        <h5>SEO Applications</h5>
                        <div class="alert alert-info">
                            <strong>Sitemap XML:</strong> Properly formatted XML sitemaps improve search engine crawling<br>
                            <strong>Schema Markup:</strong> Clean XML structure enhances structured data implementation<br>
                            <strong>Product Feeds:</strong> Well-formatted XML feeds ensure better data processing<br>
                            <strong>API Responses:</strong> Formatted XML improves API readability and debugging<br>
                            <strong>Data Exchange:</strong> Consistent XML formatting facilitates data interoperability
                        </div>
                    </div>
                    <div class="mt-4">
                        <h5>Quick Actions</h5>
                        <div class="d-flex gap-2 flex-wrap">
                            <button class="btn btn-primary" onclick="validateXml()">
                                Re-validate XML
                            </button>
                            <button class="btn btn-outline-primary" onclick="minifyXml()">
                                Create Minified Version
                            </button>
                            <button class="btn btn-outline-primary" onclick="beautifyXml()">
                                Create Beautified Version
                            </button>
                            <button class="btn btn-outline-secondary" onclick="clearResults()">
                                Clear Results
                            </button>
                        </div>
                    </div>
                `;

                resultsDiv.innerHTML = mockResults;
                document.getElementById('resultsSection').style.display = 'block';
            }, 1500);
        });

        function formatXml(xml, indentType, sortAttributes) {
            // Simple XML formatting function
            let formatted = '';
            let indent = 0;
            const indentString = indentType === 'tabs' ? '\t' : (indentType === 'spaces4' ? '    ' : '  ');

            // Basic formatting - this is a simplified version
            xml = xml.replace(/></g, '>\n<');
            const lines = xml.split('\n');

            lines.forEach(line => {
                line = line.trim();
                if (!line) return;

                if (line.startsWith('</')) {
                    indent--;
                }

                formatted += indentString.repeat(Math.max(0, indent)) + line + '\n';

                if (line.startsWith('<') && !line.startsWith('</') && !line.endsWith('/>') && !line.includes('?>')) {
                    indent++;
                }
            });

            return formatted.trim();
        }

        function countXmlElements(xml) {
            const matches = xml.match(/<[^/?!][^>]*>/g);
            return matches ? matches.length : 0;
        }

        function countXmlAttributes(xml) {
            const matches = xml.match(/\w+="[^"]*"/g);
            return matches ? matches.length : 0;
        }

        function copyFormattedXml(xml) {
            navigator.clipboard.writeText(xml).then(function() {
                alert('Formatted XML copied to clipboard!');
            });
        }

        function downloadFormattedXml(xml) {
            const blob = new Blob([xml], { type: 'application/xml' });
            const url = URL.createObjectURL(blob);
            const a = document.createElement('a');
            a.href = url;
            a.download = 'formatted-xml.xml';
            document.body.appendChild(a);
            a.click();
            document.body.removeChild(a);
            URL.revokeObjectURL(url);
        }

        function validateXml() {
            const xmlInput = document.getElementById('xmlInput').value;
            document.getElementById('formatType').value = 'validate';
            document.getElementById('xmlFormatterForm').dispatchEvent(new Event('submit'));
        }

        function minifyXml() {
            const xmlInput = document.getElementById('xmlInput').value;
            document.getElementById('formatType').value = 'minify';
            document.getElementById('xmlFormatterForm').dispatchEvent(new Event('submit'));
        }

        function beautifyXml() {
            const xmlInput = document.getElementById('xmlInput').value;
            document.getElementById('formatType').value = 'beautify';
            document.getElementById('xmlFormatterForm').dispatchEvent(new Event('submit'));
        }

        function clearResults() {
            document.getElementById('resultsSection').style.display = 'none';
            document.getElementById('xmlInput').value = '';
        }
    </script>

<?php include __DIR__ . '/../../includes/footer.php'; ?>