<?php
$page_title = "URL Encoder - Loganix SEO Agency";
include_once __DIR__ . '/../../includes/config.php';
include_once __DIR__ . '/../../includes/header.php';
?>

    <!-- JSON-LD Schema Markup -->
    <script type="application/ld+json">
    {
        "@context": "https://schema.org",
        "@type": "WebApplication",
        "name": "URL Encoder",
        "description": "Encode URLs and text for safe transmission over the internet and web browsers with proper character escaping.",
        "url": "<?php echo $base_url; ?>/tools/url-encoder",
        "applicationCategory": "DeveloperApplication",
        "operatingSystem": "Web Browser",
        "offers": {
            "@type": "Offer",
            "price": "0",
            "priceCurrency": "USD"
        },
        "provider": {
            "@type": "Organization",
            "name": "Loganix SEO Agency",
            "url": "<?php echo $base_url; ?>"
        },
        "featureList": [
            "URL encoding for safe transmission",
            "Special character handling",
            "Browser compatibility",
            "Real-time encoding",
            "Copy and download functionality"
        ]
    }
    </script>

    <!-- Tool Hero Section -->
    <section class="hero-section" style="background: linear-gradient(135deg, #0153C8 0%, #092D57 100%);">
        <div class="container">
            <div class="row align-items-center">
                <div class="col-12 text-center hero-content">
                    <h1 class="fade-in" style="font-size: 2.5rem;">URL Encoder</h1>
                    <p class="fade-in">Encode URLs and text for safe transmission over the internet and web browsers with proper character escaping.</p>
                </div>
            </div>
        </div>
    </section>

    <!-- Tool Interface -->
    <section class="py-5">
        <div class="container">
            <div class="row">
                <div class="col-lg-10 mx-auto">
                    <div class="service-card p-4">
                        <h3 class="mb-4">Encode URL or Text</h3>
                        <form id="urlEncoderForm">
                            <div class="mb-3">
                                <label for="inputText" class="form-label">Text to Encode</label>
                                <textarea class="form-control" id="inputText" rows="4" placeholder="Enter URL or text to encode..." required></textarea>
                            </div>
                            <div class="mb-3">
                                <label for="encodingType" class="form-label">Encoding Type</label>
                                <select class="form-control" id="encodingType">
                                    <option value="url">URL Encoding (RFC 3986)</option>
                                    <option value="component">URL Component Encoding</option>
                                    <option value="full">Full URI Encoding</option>
                                </select>
                            </div>
                            <div class="mb-3">
                                <div class="form-check">
                                    <input class="form-check-input" type="checkbox" id="encodeSpaces">
                                    <label class="form-check-label" for="encodeSpaces">
                                        Encode spaces as %20 (instead of +)
                                    </label>
                                </div>
                            </div>
                            <button type="submit" class="btn btn-primary">Encode Text</button>
                        </form>
                    </div>

                    <!-- Results Section -->
                    <div id="resultsSection" class="mt-4" style="display: none;">
                        <div class="service-card p-4">
                            <h4>URL Encoding Results</h4>
                            <div id="urlResults" class="mt-4"></div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </section>

    <!-- Article Section -->
    <section class="py-5 bg-light">
        <div class="container">
            <div class="row">
                <div class="col-lg-10 mx-auto">
                    <article>
                        <h2>The Complete Guide to URL Encoding for Web Development and SEO</h2>

                        <p>URL encoding is essential for web development and data transmission, ensuring that special characters are properly handled when sending data over the internet. A URL encoder helps convert text and URLs into a safe format that browsers and servers can understand. This comprehensive guide explores URL encoding techniques, standards, and practical applications.</p>

                        <h3>What is URL Encoding?</h3>

                        <p>URL encoding, also known as percent-encoding, is a mechanism for encoding information in a Uniform Resource Identifier (URI) under certain circumstances. It converts characters that are not allowed in a URL into a format that can be transmitted over the internet.</p>

                        <h4>Why URL Encoding Matters</h4>

                        <ul>
                            <li><strong>Safe Transmission:</strong> Ensures data integrity during HTTP requests</li>
                            <li><strong>Browser Compatibility:</strong> Handles special characters across different browsers</li>
                            <li><strong>Server Processing:</strong> Allows proper parsing of URLs and form data</li>
                            <li><strong>SEO Preservation:</strong> Maintains URL structure for search engines</li>
                        </ul>

                        <h3>URL Encoding Standards</h3>

                        <h4>RFC 3986 (URL Encoding)</h4>

                        <p>The standard for URI encoding defines which characters need to be percent-encoded:</p>

                        <ul>
                            <li>Reserved characters: : / ? # [ ] @</li>
                            <li>Unreserved characters: A-Z a-z 0-9 - . _ ~</li>
                            <li>All other characters must be encoded</li>
                        </ul>

                        <h4>URL Component Encoding</h4>

                        <p>Used for encoding individual components of a URL:</p>

                        <ul>
                            <li>Query parameters</li>
                            <li>Fragment identifiers</li>
                            <li>Path segments</li>
                        </ul>

                        <h3>Common Characters and Their Encodings</h3>

                        <table class="table table-striped">
                            <thead>
                                <tr>
                                    <th>Character</th>
                                    <th>Encoded</th>
                                    <th>Use Case</th>
                                </tr>
                            </thead>
                            <tbody>
                                <tr>
                                    <td>Space</td>
                                    <td>%20 or +</td>
                                    <td>Text with spaces</td>
                                </tr>
                                <tr>
                                    <td>&</td>
                                    <td>%26</td>
                                    <td>Query parameters</td>
                                </tr>
                                <tr>
                                    <td>=</td>
                                    <td>%3D</td>
                                    <td>Key-value pairs</td>
                                </tr>
                                <tr>
                                    <td>%</td>
                                    <td>%25</td>
                                    <td>Percent signs</td>
                                </tr>
                                <tr>
                                    <td>+</td>
                                    <td>%2B</td>
                                    <td>Plus signs</td>
                                </tr>
                                <tr>
                                    <td>?</td>
                                    <td>%3F</td>
                                    <td>Query strings</td>
                                </tr>
                            </tbody>
                        </table>

                        <h3>How URL Encoders Work</h3>

                        <p>URL encoding tools systematically convert characters to their encoded equivalents:</p>

                        <ol>
                            <li><strong>Character Analysis:</strong> Identify characters that need encoding</li>
                            <li><strong>UTF-8 Conversion:</strong> Convert characters to UTF-8 byte sequences</li>
                            <li><strong>Percent Encoding:</strong> Apply %XX format for each byte</li>
                            <li><strong>Context Handling:</strong> Apply appropriate encoding based on context</li>
                            <li><strong>Validation:</strong> Ensure proper encoding format</li>
                        </ol>

                        <h3>URL Encoding in Different Contexts</h3>

                        <h4>Query Parameters</h4>

                        <p>Encoding data sent in URL query strings:</p>

                        <pre><code>Original: https://example.com/search?q=hello world&category=web dev
Encoded:  https://example.com/search?q=hello%20world&category=web%20dev</code></pre>

                        <h4>Form Data</h4>

                        <p>Encoding form submissions:</p>

                        <pre><code>POST data: name=John Doe&email=john@example.com
Encoded:  name=John%20Doe&email=john%40example.com</code></pre>

                        <h4>API Endpoints</h4>

                        <p>Encoding API parameters and paths:</p>

                        <pre><code>API call: /api/users/123/profile?name=John Paul
Encoded:  /api/users/123/profile?name=John%20Paul</code></pre>

                        <h3>URL Encoding Best Practices</h3>

                        <h4>When to Encode</h4>

                        <ul>
                            <li>Always encode data before sending in URLs</li>
                            <li>Encode user input to prevent injection attacks</li>
                            <li>Encode special characters in file names</li>
                            <li>Encode non-ASCII characters</li>
                        </ul>

                        <h4>When Not to Encode</h4>

                        <ul>
                            <li>Don't double-encode already encoded strings</li>
                            <li>Avoid encoding already valid URL components</li>
                            <li>Don't encode the main URL structure (protocol, domain)</li>
                        </ul>

                        <h3>URL Encoding and SEO</h3>

                        <h4>URL Structure Preservation</h4>

                        <p>Proper encoding maintains URL readability for SEO:</p>

                        <ul>
                            <li>Keep URLs clean and descriptive</li>
                            <li>Avoid over-encoding that hurts readability</li>
                            <li>Use encoding strategically for special characters</li>
                        </ul>

                        <h4>International SEO</h4>

                        <p>Encoding non-ASCII characters in URLs:</p>

                        <ul>
                            <li>Proper encoding for international domains</li>
                            <li>Support for Unicode characters</li>
                            <li>Browser compatibility across languages</li>
                        </ul>

                        <h3>Common URL Encoding Issues</h3>

                        <h4>Double Encoding</h4>

                        <p>Encoding already encoded strings:</p>

                        <pre><code>Wrong: hello%2520world (space encoded twice)
Right: hello%20world (space encoded once)</code></pre>

                        <h4>Incorrect Context</h4>

                        <p>Using wrong encoding for the context:</p>

                        <ul>
                            <li>Using URI encoding where component encoding is needed</li>
                            <li>Mixing encoding standards inappropriately</li>
                        </ul>

                        <h3>URL Encoding Tools and Libraries</h3>

                        <p>Various tools help with URL encoding:</p>

                        <ul>
                            <li><strong>Browser APIs:</strong> encodeURI(), encodeURIComponent()</li>
                            <li><strong>Server Libraries:</strong> PHP urlencode(), Python urllib</li>
                            <li><strong>Online Tools:</strong> Web-based encoding utilities</li>
                            <li><strong>Developer Tools:</strong> Browser console encoding functions</li>
                        </ul>

                        <h3>Advanced URL Encoding Topics</h3>

                        <h4>Base64 URL Encoding</h4>

                        <p>URL-safe Base64 encoding for tokens:</p>

                        <pre><code>Standard Base64: ABC+DEF/GHI=
URL-safe Base64: ABC-DEF_GHI</code></pre>

                        <h4>Unicode Handling</h4>

                        <p>Encoding Unicode characters:</p>

                        <ul>
                            <li>UTF-8 encoding for international characters</li>
                            <li>Proper handling of emojis and symbols</li>
                            <li>Browser support for Unicode in URLs</li>
                        </ul>

                        <h3>URL Encoding in Web Development</h3>

                        <h4>JavaScript Encoding</h4>

                        <pre><code>// For entire URLs
encodeURI("https://example.com/path with spaces")

// For URL components
encodeURIComponent("hello world & more")</code></pre>

                        <h4>PHP Encoding</h4>

                        <pre><code>// URL encoding
urlencode("hello world");

// Raw URL encoding
rawurlencode("hello world");</code></pre>

                        <h4>Python Encoding</h4>

                        <pre><code>from urllib.parse import quote

# URL encoding
quote("hello world & more")</code></pre>

                        <h3>Measuring Encoding Effectiveness</h3>

                        <p>Track encoding performance and correctness:</p>

                        <ul>
                            <li><strong>Encoding Accuracy:</strong> Proper character conversion</li>
                            <li><strong>Decoding Compatibility:</strong> Successful round-trip encoding/decoding</li>
                            <li><strong>Performance Impact:</strong> Encoding/decoding speed</li>
                            <li><strong>Cross-Platform Support:</strong> Compatibility across systems</li>
                        </ul>

                        <h3>Future of URL Encoding</h3>

                        <p>URL encoding continues to evolve:</p>

                        <ul>
                            <li><strong>HTTP/2:</strong> Header compression reduces encoding needs</li>
                            <li><strong>Unicode URLs:</strong> Better support for international characters</li>
                            <li><strong>Binary Protocols:</strong> Alternative data transmission methods</li>
                        </ul>

                        <h3>Conclusion</h3>

                        <p>URL encoding is a fundamental aspect of web development that ensures safe and reliable data transmission. A URL encoder helps convert text and special characters into a format that can be safely sent over the internet. By understanding encoding standards and best practices, you can ensure your web applications handle data correctly and maintain compatibility across different systems.</p>

                        <p>Remember that proper URL encoding is crucial for security, data integrity, and cross-platform compatibility. Always encode user input and special characters before including them in URLs or transmitting them over the web.</p>

                        <p>Combine URL encoding with other web development tools like our <a href="<?php echo $base_url; ?>/tools/url-decoder">URL decoder</a> and <a href="<?php echo $base_url; ?>/tools/base64-encoder">Base64 encoder</a> for comprehensive data handling.</p>

                        <p>For more information on URL encoding standards, check the <a href="https://tools.ietf.org/html/rfc3986" target="_blank" rel="noopener">RFC 3986 specification</a> and <a href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/encodeURI" target="_blank" rel="noopener">MDN Web Docs on encoding</a>. Start encoding your URLs safely today and improve your web application's reliability.</p>
                    </article>
                </div>
            </div>
        </div>
    </section>

    <!-- Related Tools -->
    <section class="py-5">
        <div class="container">
            <div class="section-title fade-in">
                <h2>Related SEO Tools</h2>
                <p>Explore our other powerful SEO analysis tools</p>
            </div>
            <div class="row">
                <div class="col-lg-4 fade-in">
                    <div class="service-card">
                        <div class="icon">🔓</div>
                        <h4>URL Decoder</h4>
                        <p>Decode URL-encoded strings back to their original readable format.</p>
                        <a href="<?php echo $base_url; ?>/tools/url-decoder" class="btn btn-primary mt-3">Use Tool</a>
                    </div>
                </div>
                <div class="col-lg-4 fade-in">
                    <div class="service-card">
                        <div class="icon">🔒</div>
                        <h4>Base64 Encoder</h4>
                        <p>Encode text and data to Base64 format for secure transmission and storage.</p>
                        <a href="<?php echo $base_url; ?>/tools/base64-encoder" class="btn btn-primary mt-3">Use Tool</a>
                    </div>
                </div>
                <div class="col-lg-4 fade-in">
                    <div class="service-card">
                        <div class="icon">🔗</div>
                        <h4>Slug Generator</h4>
                        <p>Generate SEO-friendly URL slugs from titles and keywords for clean web addresses.</p>
                        <a href="<?php echo $base_url; ?>/tools/slug-generator" class="btn btn-primary mt-3">Use Tool</a>
                    </div>
                </div>
            </div>
        </div>
    </section>

    <!-- CTA Section -->
    <section class="cta-section">
        <div class="container text-center fade-in">
            <h2>Need Professional SEO Services?</h2>
            <p>While our tools are great for analysis, our expert team can help you implement advanced SEO strategies.</p>
            <a href="<?php echo $base_url; ?>/pages/contact" class="btn btn-light btn-lg">Get Expert Help</a>
        </div>
    </section>

    <script>
        document.getElementById('urlEncoderForm').addEventListener('submit', function(e) {
            e.preventDefault();

            const inputText = document.getElementById('inputText').value;
            const encodingType = document.getElementById('encodingType').value;
            const encodeSpaces = document.getElementById('encodeSpaces').checked;
            const resultsDiv = document.getElementById('urlResults');

            if (!inputText.trim()) {
                alert('Please enter text to encode.');
                return;
            }

            resultsDiv.innerHTML = '<div class="text-center"><div class="spinner-border" role="status"><span class="visually-hidden">Loading...</span></div><p class="mt-2">Encoding text...</p></div>';

            // In a real implementation, this would encode the text
            // For demo purposes, we'll simulate the results
            setTimeout(() => {
                let encodedText = '';

                switch(encodingType) {
                    case 'url':
                        encodedText = encodeURI(inputText);
                        break;
                    case 'component':
                        encodedText = encodeURIComponent(inputText);
                        break;
                    case 'full':
                        encodedText = encodeURIComponent(inputText);
                        break;
                }

                // Handle space encoding preference
                if (!encodeSpaces && encodingType === 'component') {
                    encodedText = encodedText.replace(/%20/g, '+');
                }

                const mockResults = `
                    <div class="alert alert-success">
                        <strong>✓ URL Encoding Complete!</strong>
                    </div>
                    <div class="row mt-4">
                        <div class="col-md-6">
                            <h5>Encoding Summary</h5>
                            <div class="alert alert-info">
                                <strong>Original Length:</strong> ${inputText.length} characters<br>
                                <strong>Encoded Length:</strong> ${encodedText.length} characters<br>
                                <strong>Encoding Type:</strong> ${encodingType.charAt(0).toUpperCase() + encodingType.slice(1)}<br>
                                <strong>Space Handling:</strong> ${encodeSpaces ? '%20' : '+'}
                            </div>
                        </div>
                        <div class="col-md-6">
                            <h5>Encoding Details</h5>
                            <div class="alert alert-success">
                                <strong>Characters Encoded:</strong> ${countEncodedChars(inputText, encodedText)}<br>
                                <strong>Encoding Efficiency:</strong> ${Math.round((encodedText.length / inputText.length) * 100)}%<br>
                                <strong>Standard Compliance:</strong> RFC 3986<br>
                                <strong>Browser Compatible:</strong> Yes
                            </div>
                        </div>
                    </div>
                    <div class="mt-4">
                        <h5>Encoded Result</h5>
                        <div class="input-group">
                            <input type="text" class="form-control" id="encodedResult" value="${encodedText}" readonly>
                            <button class="btn btn-outline-primary" type="button" onclick="copyEncodedText()">
                                <i class="fas fa-copy"></i> Copy
                            </button>
                        </div>
                        <div class="mt-2">
                            <small class="text-muted">Safe to use in URLs, forms, and API calls</small>
                        </div>
                    </div>
                    <div class="mt-4">
                        <h5>Before & After Comparison</h5>
                        <div class="row">
                            <div class="col-md-6">
                                <h6>Original Text</h6>
                                <div class="bg-light p-3 rounded" style="font-family: 'Courier New', monospace; font-size: 14px; max-height: 150px; overflow-y: auto;">
                                    ${inputText.replace(/&/g, '&').replace(/</g, '<').replace(/>/g, '>')}
                                </div>
                            </div>
                            <div class="col-md-6">
                                <h6>Encoded Text</h6>
                                <div class="bg-light p-3 rounded" style="font-family: 'Courier New', monospace; font-size: 14px; max-height: 150px; overflow-y: auto;">
                                    ${encodedText}
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="mt-4">
                        <h5>Character Encoding Analysis</h5>
                        <div class="table-responsive">
                            <table class="table table-striped">
                                <thead>
                                    <tr>
                                        <th>Character</th>
                                        <th>Encoded As</th>
                                        <th>Reason</th>
                                    </tr>
                                </thead>
                                <tbody id="encodingTable">
                                    ${generateEncodingTable(inputText, encodedText)}
                                </tbody>
                            </table>
                        </div>
                    </div>
                    <div class="mt-4">
                        <h5>Usage Examples</h5>
                        <div class="bg-light p-3 rounded">
                            <small>
                                <strong>Query Parameter:</strong> https://example.com/search?q=${encodedText}<br>
                                <strong>Form Data:</strong> name=${encodedText}&email=user@example.com<br>
                                <strong>API Call:</strong> /api/search/${encodedText}<br>
                                <strong>URL Fragment:</strong> https://example.com/page#${encodedText}
                            </small>
                        </div>
                    </div>
                    <div class="mt-4">
                        <h5>SEO & Security Benefits</h5>
                        <ul class="list-group">
                            <li class="list-group-item list-group-item-success">✓ Prevents URL injection attacks</li>
                            <li class="list-group-item list-group-item-success">✓ Ensures cross-browser compatibility</li>
                            <li class="list-group-item list-group-item-success">✓ Maintains data integrity in transmission</li>
                            <li class="list-group-item list-group-item-info">ℹ Follows RFC 3986 encoding standards</li>
                            <li class="list-group-item list-group-item-info">ℹ Safe for use in GET and POST requests</li>
                            <li class="list-group-item list-group-item-success">✓ Protects against special character conflicts</li>
                        </ul>
                    </div>
                    <div class="mt-4">
                        <h5>Quick Actions</h5>
                        <div class="d-flex gap-2 flex-wrap">
                            <button class="btn btn-primary" onclick="testEncodedURL()">
                                Test Encoded URL
                            </button>
                            <button class="btn btn-outline-primary" onclick="downloadEncodedText()">
                                Download as File
                            </button>
                            <button class="btn btn-outline-primary" onclick="clearResults()">
                                Clear Results
                            </button>
                        </div>
                    </div>
                `;

                resultsDiv.innerHTML = mockResults;
                document.getElementById('resultsSection').style.display = 'block';
            }, 1000);
        });

        function countEncodedChars(original, encoded) {
            // Count characters that were encoded
            let encodedCount = 0;
            for (let i = 0; i < encoded.length; i++) {
                if (encoded[i] === '%' && i + 2 < encoded.length) {
                    if (/^[0-9A-Fa-f]{2}$/.test(encoded.substr(i + 1, 2))) {
                        encodedCount++;
                        i += 2; // Skip the next two characters
                    }
                }
            }
            return encodedCount;
        }

        function generateEncodingTable(original, encoded) {
            const encodingMap = {};
            let i = 0, j = 0;

            while (i < original.length && j < encoded.length) {
                if (original[i] === encoded[j]) {
                    i++;
                    j++;
                } else if (encoded.substr(j, 3).match(/^%[0-9A-Fa-f]{2}$/)) {
                    encodingMap[original[i]] = encoded.substr(j, 3);
                    i++;
                    j += 3;
                } else {
                    i++;
                    j++;
                }
            }

            let tableRows = '';
            for (const [char, encoded] of Object.entries(encodingMap)) {
                const reason = getEncodingReason(char);
                tableRows += `<tr><td>'${char}'</td><td>${encoded}</td><td>${reason}</td></tr>`;
            }

            return tableRows || '<tr><td colspan="3" class="text-center">No special characters required encoding</td></tr>';
        }

        function getEncodingReason(char) {
            const reasons = {
                ' ': 'Space character',
                '&': 'Reserved character (&)',
                '=': 'Reserved character (=)',
                '?': 'Reserved character (?)',
                '#': 'Reserved character (#)',
                '%': 'Percent sign (%)',
                '+': 'Plus sign (+)',
                '<': 'Less than (<)',
                '>': 'Greater than (>)',
                '"': 'Quote (")',
                "'": 'Apostrophe (')',
                '{': 'Curly brace ({)',
                '}': 'Curly brace (})',
                '|': 'Pipe (|)',
                '\\': 'Backslash (\\)',
                '^': 'Caret (^)',
                '~': 'Tilde (~)',
                '[': 'Square bracket ([)',
                ']': 'Square bracket (])',
                '`': 'Backtick (`)'
            };
            return reasons[char] || 'Special/reserved character';
        }

        function copyEncodedText() {
            const encodedResult = document.getElementById('encodedResult');
            encodedResult.select();
            document.execCommand('copy');

            // Show feedback
            const button = event.target.closest('button');
            const originalHTML = button.innerHTML;
            button.innerHTML = '<i class="fas fa-check"></i> Copied!';
            button.classList.remove('btn-outline-primary');
            button.classList.add('btn-success');

            setTimeout(() => {
                button.innerHTML = originalHTML;
                button.classList.remove('btn-success');
                button.classList.add('btn-outline-primary');
            }, 2000);
        }

        function testEncodedURL() {
            const encodedText = document.getElementById('encodedResult').value;
            const testURL = `https://example.com/test?q=${encodedText}`;
            window.open(testURL, '_blank');
        }

        function downloadEncodedText() {
            const encodedText = document.getElementById('encodedResult').value;
            const blob = new Blob([encodedText], { type: 'text/plain' });
            const url = URL.createObjectURL(blob);
            const a = document.createElement('a');
            a.href = url;
            a.download = 'encoded-text.txt';
            document.body.appendChild(a);
            a.click();
            document.body.removeChild(a);
            URL.revokeObjectURL(url);
        }

        function clearResults() {
            document.getElementById('resultsSection').style.display = 'none';
            document.getElementById('inputText').value = '';
        }
    </script>

<?php include __DIR__ . '/../../includes/footer.php'; ?>