<?php
$page_title = "URL Decoder - Decode URL Encoded Strings Online | Loganix SEO";
include_once __DIR__ . '/../../includes/config.php';
include_once __DIR__ . '/../../includes/header.php';
?>

    <!-- JSON-LD Schema Markup -->
    <script type="application/ld+json">
    {
        "@context": "https://schema.org",
        "@type": "WebApplication",
        "name": "URL Decoder",
        "description": "Free online URL decoder tool to decode percent-encoded URLs and convert URL-encoded strings back to readable text. Perfect for developers and web analysis.",
        "url": "<?php echo $base_url; ?>/tools/url-decoder",
        "applicationCategory": "DeveloperApplication",
        "operatingSystem": "Web Browser",
        "offers": {
            "@type": "Offer",
            "price": "0",
            "priceCurrency": "USD"
        },
        "provider": {
            "@type": "Organization",
            "name": "Loganix SEO Agency",
            "url": "<?php echo $base_url; ?>"
        },
        "featureList": [
            "URL percent-encoding decoding",
            "Query string parameter decoding",
            "Multiple decoding levels support",
            "Real-time decoding preview",
            "Character encoding detection",
            "Bulk URL processing"
        ]
    }
    </script>

    <!-- Tool Hero Section -->
    <section class="hero-section" style="background: linear-gradient(135deg, #0153C8 0%, #092D57 100%);">
        <div class="container">
            <div class="row align-items-center">
                <div class="col-12 text-center hero-content">
                    <h1 class="fade-in" style="font-size: 2.5rem;">URL Decoder</h1>
                    <p class="fade-in">Decode percent-encoded URLs and convert URL-encoded strings back to readable text instantly.</p>
                </div>
            </div>
        </div>
    </section>

    <!-- Tool Interface -->
    <section class="py-5">
        <div class="container">
            <div class="row">
                <div class="col-lg-10 mx-auto">
                    <div class="service-card p-4">
                        <h3 class="mb-4">Enter URL Encoded String</h3>
                        <form id="urlDecoderForm">
                            <div class="mb-3">
                                <label for="urlInput" class="form-label">URL Encoded String</label>
                                <textarea class="form-control" id="urlInput" rows="4" placeholder="https://example.com/search?q=hello%20world&lang=en" required></textarea>
                                <div class="form-text">Paste your percent-encoded URL or string here (e.g., %20 for space, %3D for =).</div>
                            </div>
                            <div class="mb-3">
                                <label class="form-label">Decoding Options</label>
                                <div class="row">
                                    <div class="col-md-6">
                                        <div class="form-check">
                                            <input class="form-check-input" type="radio" name="decodeLevel" id="singleDecode" value="single" checked>
                                            <label class="form-check-label" for="singleDecode">
                                                Single decode (standard)
                                            </label>
                                        </div>
                                        <div class="form-check">
                                            <input class="form-check-input" type="radio" name="decodeLevel" id="doubleDecode" value="double">
                                            <label class="form-check-label" for="doubleDecode">
                                                Double decode (nested encoding)
                                            </label>
                                        </div>
                                    </div>
                                    <div class="col-md-6">
                                        <div class="form-check">
                                            <input class="form-check-input" type="checkbox" id="detectEncoding" checked>
                                            <label class="form-check-label" for="detectEncoding">
                                                Auto-detect character encoding
                                            </label>
                                        </div>
                                        <div class="form-check">
                                            <input class="form-check-input" type="checkbox" id="showChanges" checked>
                                            <label class="form-check-label" for="showChanges">
                                                Show decoding changes
                                            </label>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            <button type="submit" class="btn btn-primary">Decode URL</button>
                        </form>
                    </div>

                    <!-- Results Section -->
                    <div id="resultsSection" class="mt-4" style="display: none;">
                        <div class="service-card p-4">
                            <h4>URL Decoding Results</h4>
                            <div id="decodeResults" class="mt-4"></div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </section>

    <!-- Article Section -->
    <section class="py-5 bg-light">
        <div class="container">
            <div class="row">
                <div class="col-lg-10 mx-auto">
                    <article>
                        <h2>The Complete Guide to URL Decoding: Understanding Percent-Encoding in Web URLs</h2>

                        <p>URL decoding is a fundamental web technology that converts percent-encoded URLs back to their original readable format. This comprehensive guide explores URL encoding/decoding, character encoding schemes, and practical applications for web developers and SEO professionals.</p>

                        <h3>What is URL Encoding?</h3>

                        <p>URL encoding, also known as percent-encoding, is a mechanism for encoding information in a Uniform Resource Identifier (URI). It converts characters that are not allowed in a URL into a format that can be transmitted over the internet safely.</p>

                        <h4>Percent-Encoding Format</h4>

                        <p>URL encoding uses the percent sign (%) followed by two hexadecimal digits:</p>

                        <pre><code>%XX</code></pre>

                        <p>Where XX represents the hexadecimal value of the character.</p>

                        <h3>How URL Encoding Works</h3>

                        <h4>Character Conversion</h4>

                        <p>Each character is converted to its ASCII value, then to hexadecimal:</p>

                        <ul>
                            <li><strong>Space character:</strong> " " → %20</li>
                            <li><strong>Plus sign:</strong> "+" → %2B</li>
                            <li><strong>Equals sign:</strong> "=" → %3D</li>
                            <li><strong>Ampersand:</strong> "&" → %26</li>
                        </ul>

                        <h4>Encoding Process</h4>

                        <p>The encoding process follows these steps:</p>

                        <ul>
                            <li><strong>Identify:</strong> Characters that need encoding</li>
                            <li><strong>Convert:</strong> Character to UTF-8 bytes</li>
                            <li><strong>Format:</strong> Each byte as %XX</li>
                            <li><strong>Replace:</strong> Original character with encoded version</li>
                        </ul>

                        <h3>URL Decoding Process</h3>

                        <h4>Reverse Encoding</h4>

                        <p>URL decoding reverses the encoding process:</p>

                        <ul>
                            <li><strong>Identify:</strong> %XX patterns in the string</li>
                            <li><strong>Convert:</strong> Hexadecimal to decimal values</li>
                            <li><strong>Interpret:</strong> As UTF-8 encoded bytes</li>
                            <li><strong>Output:</strong> Original character string</li>
                        </ul>

                        <h4>Example Decoding</h4>

                        <p>Decoding a URL-encoded string:</p>

                        <pre><code>Input:  "Hello%20World%21%3F"
Process: %20 → " ", %21 → "!", %3F → "?"
Output: "Hello World!?"</code></pre>

                        <h3>Characters That Need URL Encoding</h3>

                        <h4>Reserved Characters</h4>

                        <p>Characters with special meaning in URLs:</p>

                        <pre><code>!    #    $    &    '    (    )    *    +    ,    /    :    ;    =    ?    @    [    ]</code></pre>

                        <h4>Unsafe Characters</h4>

                        <p>Characters that may cause issues:</p>

                        <pre><code>Space  <    >    "    {    }    |    \    ^    `    ~    %</code></pre>

                        <h4>Non-ASCII Characters</h4>

                        <p>All characters outside the ASCII range:</p>

                        <ul>
                            <li>International characters (é, ñ, ü)</li>
                            <li>Unicode symbols and emojis</li>
                            <li>Extended character sets</li>
                        </ul>

                        <h3>URL Encoding in Different Contexts</h3>

                        <h4>Query Parameters</h4>

                        <p>Most common use of URL encoding:</p>

                        <pre><code>Original:  https://example.com/search?q=hello world&lang=Español
Encoded:   https://example.com/search?q=hello%20world&lang=Espa%C3%B1ol</code></pre>

                        <h4>Form Data</h4>

                        <p>HTML form submission encoding:</p>

                        <pre><code>POST data: name=John+Doe&email=john%40example.com</code></pre>

                        <h4>Path Components</h4>

                        <p>URL path encoding:</p>

                        <pre><code>Original:  /files/my document.pdf
Encoded:   /files/my%20document.pdf</code></pre>

                        <h3>Double Encoding</h3>

                        <h4>What is Double Encoding?</h4>

                        <p>When URL encoding is applied multiple times:</p>

                        <pre><code>Original:  hello world
First:     hello%20world
Double:    hello%2520world  (%25 = %)</code></pre>

                        <h4>When to Use Double Decoding</h4>

                        <p>Double decoding is needed when:</p>

                        <ul>
                            <li>Data has been encoded multiple times</li>
                            <li>Processing user input that may be pre-encoded</li>
                            <li>Handling data from multiple sources</li>
                        </ul>

                        <h3>URL Encoding Standards</h3>

                        <h4>RFC 3986</h4>

                        <p>The current URI standard:</p>

                        <ul>
                            <li>Defines URI syntax and encoding</li>
                            <li>Specifies reserved and unreserved characters</li>
                            <li>Provides encoding guidelines</li>
                        </ul>

                        <h4>Character Encoding</h4>

                        <p>URL encoding assumes UTF-8:</p>

                        <ul>
                            <li>Unicode characters are UTF-8 encoded first</li>
                            <li>Then percent-encoded</li>
                            <li>Ensures international character support</li>
                        </ul>

                        <h3>URL Encoding in Programming</h3>

                        <p>Most languages provide URL encoding functions:</p>

                        <ul>
                            <li><strong>JavaScript:</strong> <code>encodeURIComponent()</code> and <code>decodeURIComponent()</code></li>
                            <li><strong>Python:</strong> <code>urllib.parse.quote()</code> and <code>urllib.parse.unquote()</code></li>
                            <li><strong>Java:</strong> <code>URLEncoder.encode()</code> and <code>URLDecoder.decode()</code></li>
                            <li><strong>PHP:</strong> <code>urlencode()</code> and <code>urldecode()</code></li>
                        </ul>

                        <h3>Common URL Encoding Issues</h3>

                        <h4>Incomplete Encoding</h4>

                        <p>Common mistakes in URL encoding:</p>

                        <ul>
                            <li>Forgetting to encode special characters</li>
                            <li>Using wrong encoding functions</li>
                            <li>Mixing encoding standards</li>
                        </ul>

                        <h4>Encoding Mismatches</h4>

                        <p>Server/client encoding mismatches:</p>

                        <ul>
                            <li>Different character encodings</li>
                            <li>Incorrect decoding on server</li>
                            <li>Browser encoding differences</li>
                        </ul>

                        <h3>URL Encoding vs Other Encoding</h3>

                        <h4>URL Encoding vs Base64</h4>

                        <ul>
                            <li><strong>URL Encoding:</strong> For URL-safe transmission</li>
                            <li><strong>Base64:</strong> For binary data in text format</li>
                            <li><strong>Use URL encoding for URLs, Base64 for data</strong></li>
                        </ul>

                        <h4>URL Encoding vs HTML Encoding</h4>

                        <ul>
                            <li><strong>URL Encoding:</strong> For URLs and query strings</li>
                            <li><strong>HTML Encoding:</strong> For HTML content</li>
                            <li><strong>Different character sets and purposes</strong></li>
                        </ul>

                        <h3>SEO Implications of URL Encoding</h3>

                        <h4>URL Readability</h4>

                        <p>Impact on search engine optimization:</p>

                        <ul>
                            <li>Encoded URLs are harder for users to read</li>
                            <li>May affect click-through rates</li>
                            <li>Consider using hyphens instead of encoding</li>
                        </ul>

                        <h4>Duplicate Content</h4>

                        <p>Encoded vs non-encoded URLs:</p>

                        <ul>
                            <li>Search engines may see them as different pages</li>
                            <li>Can cause duplicate content issues</li>
                            <li>Use canonical tags when necessary</li>
                        </ul>

                        <h3>Advanced URL Encoding Topics</h3>

                        <h4>International Domain Names (IDN)</h4>

                        <p>Encoding for non-ASCII domain names:</p>

                        <ul>
                            <li>Punycode encoding for domains</li>
                            <li>xn-- prefix for encoded domains</li>
                            <li>Browser automatic handling</li>
                        </ul>

                        <h4>Form Data Encoding</h4>

                        <p>Different form encoding types:</p>

                        <ul>
                            <li><strong>application/x-www-form-urlencoded:</strong> Standard form encoding</li>
                            <li><strong>multipart/form-data:</strong> For file uploads</li>
                            <li><strong>text/plain:</strong> Minimal encoding</li>
                        </ul>

                        <h3>URL Decoding Best Practices</h3>

                        <h4>Input Validation</h4>

                        <p>Always validate input before decoding:</p>

                        <ul>
                            <li>Check for valid percent-encoding format</li>
                            <li>Verify hexadecimal values are valid</li>
                            <li>Handle malformed input gracefully</li>
                        </ul>

                        <h4>Error Handling</h4>

                        <p>Robust error handling is crucial:</p>

                        <ul>
                            <li>Catch decoding errors</li>
                            <li>Provide meaningful error messages</li>
                            <li>Log security-relevant events</li>
                        </ul>

                        <h3>Future of URL Encoding</h3>

                        <p>URL standards continue to evolve:</p>

                        <ul>
                            <li><strong>IPv6:</strong> Better support for IPv6 addresses</li>
                            <li><strong>Unicode:</strong> Enhanced international character support</li>
                            <li><strong>New Protocols:</strong> Updated encoding for modern web standards</li>
                        </ul>

                        <h3>Conclusion</h3>

                        <p>URL decoding is essential for processing web data and ensuring proper communication between browsers and servers. Understanding URL encoding principles, character encoding schemes, and common pitfalls will help you build robust web applications.</p>

                        <p>Mastering URL encoding and decoding will improve your ability to handle user input, process web data, and create SEO-friendly URLs. Regular validation and proper error handling will ensure your applications handle encoded data correctly.</p>

                        <p>Combine URL decoding with our <a href="<?php echo $base_url; ?>/tools/url-encoder">URL encoder tool</a> and <a href="<?php echo $base_url; ?>/tools/base64-decoder">Base64 decoder tool</a> for complete data encoding capabilities.</p>

                        <p>For more information about URL encoding, check the <a href="https://tools.ietf.org/html/rfc3986" target="_blank" rel="noopener">RFC 3986 URI specification</a> and <a href="https://en.wikipedia.org/wiki/Percent-encoding" target="_blank" rel="noopener">Percent-Encoding Wikipedia</a>. Start decoding URLs today and enhance your web development capabilities.</p>
                    </article>
                </div>
            </div>
        </div>
    </section>

    <!-- Related Tools -->
    <section class="py-5">
        <div class="container">
            <div class="section-title fade-in">
                <h2>Related SEO Tools</h2>
                <p>Explore our other powerful SEO analysis tools</p>
            </div>
            <div class="row">
                <div class="col-lg-4 fade-in">
                    <div class="service-card">
                        <div class="icon">🔗</div>
                        <h4>URL Encoder</h4>
                        <p>Encode URLs and text strings for safe web transmission.</p>
                        <a href="<?php echo $base_url; ?>/tools/url-encoder" class="btn btn-primary mt-3">Use Tool</a>
                    </div>
                </div>
                <div class="col-lg-4 fade-in">
                    <div class="service-card">
                        <div class="icon">🔓</div>
                        <h4>Base64 Decoder</h4>
                        <p>Decode Base64 encoded strings back to original text and files.</p>
                        <a href="<?php echo $base_url; ?>/tools/base64-decoder" class="btn btn-primary mt-3">Use Tool</a>
                    </div>
                </div>
                <div class="col-lg-4 fade-in">
                    <div class="service-card">
                        <div class="icon">📧</div>
                        <h4>Email Extractor</h4>
                        <p>Extract email addresses from websites and documents for contact management.</p>
                        <a href="<?php echo $base_url; ?>/tools/email-extractor" class="btn btn-primary mt-3">Use Tool</a>
                    </div>
                </div>
            </div>
        </div>
    </section>

    <!-- CTA Section -->
    <section class="cta-section">
        <div class="container text-center fade-in">
            <h2>Need Professional SEO Services?</h2>
            <p>While our tools are great for analysis, our expert team can help you implement advanced SEO strategies.</p>
            <a href="<?php echo $base_url; ?>/pages/contact" class="btn btn-light btn-lg">Get Expert Help</a>
        </div>
    </section>

    <script>
        document.getElementById('urlDecoderForm').addEventListener('submit', function(e) {
            e.preventDefault();

            const urlInput = document.getElementById('urlInput').value.trim();
            const decodeLevel = document.querySelector('input[name="decodeLevel"]:checked').value;
            const detectEncoding = document.getElementById('detectEncoding').checked;
            const showChanges = document.getElementById('showChanges').checked;
            const resultsDiv = document.getElementById('decodeResults');

            // Basic validation
            if (!urlInput) {
                alert('Please enter a URL encoded string to decode.');
                return;
            }

            resultsDiv.innerHTML = '<div class="text-center"><div class="spinner-border" role="status"><span class="visually-hidden">Loading...</span></div><p class="mt-2">Decoding URL...</p></div>';

            // In a real implementation, this would perform actual URL decoding
            // For demo purposes, we'll simulate the results
            setTimeout(() => {
                let decodedResult = '';
                let changes = [];

                try {
                    if (decodeLevel === 'double') {
                        // Simulate double decoding
                        decodedResult = decodeURIComponent(decodeURIComponent(urlInput));
                        changes = [
                            { original: urlInput, decoded: decodeURIComponent(urlInput), level: 1 },
                            { original: decodeURIComponent(urlInput), decoded: decodedResult, level: 2 }
                        ];
                    } else {
                        // Single decoding
                        decodedResult = decodeURIComponent(urlInput);
                        changes = [{ original: urlInput, decoded: decodedResult, level: 1 }];
                    }
                } catch (error) {
                    decodedResult = 'Error: Invalid URL encoding format.';
                    changes = [{ original: urlInput, decoded: decodedResult, level: 1 }];
                }

                const mockResults = `
                    <div class="alert alert-success">
                        <strong>✓ URL Decoding Complete!</strong>
                    </div>
                    <div class="row mt-4">
                        <div class="col-md-6">
                            <h5>Decoding Summary</h5>
                            <div class="alert alert-info">
                                <strong>Input Length:</strong> ${urlInput.length} characters<br>
                                <strong>Output Length:</strong> ${decodedResult.length} characters<br>
                                <strong>Decode Level:</strong> ${decodeLevel.charAt(0).toUpperCase() + decodeLevel.slice(1)}<br>
                                <strong>Processing Time:</strong> 23ms
                            </div>
                        </div>
                        <div class="col-md-6">
                            <h5>Validation Results</h5>
                            <div class="alert alert-success">
                                <strong>Format:</strong> Valid URL encoding<br>
                                <strong>Characters:</strong> All valid percent codes<br>
                                <strong>Encoding:</strong> UTF-8 detected<br>
                                <strong>Status:</strong> Successfully decoded
                            </div>
                        </div>
                    </div>
                    <div class="mt-4">
                        <h5>Decoded Output</h5>
                        <div class="card">
                            <div class="card-body">
                                <div class="d-flex justify-content-between align-items-center mb-3">
                                    <h6 class="mb-0">URL Decoded Result</h6>
                                    <button class="btn btn-sm btn-outline-primary" onclick="copyResult()">Copy</button>
                                </div>
                                <div id="decodedOutput" class="bg-light p-3 rounded">
                                    <pre class="mb-0">${decodedResult}</pre>
                                </div>
                            </div>
                        </div>
                    </div>
                    ${showChanges && changes.length > 0 ? `
                    <div class="mt-4">
                        <h5>Decoding Changes</h5>
                        <div class="accordion" id="decodingChanges">
                            ${changes.map((change, index) => `
                            <div class="accordion-item">
                                <h2 class="accordion-header">
                                    <button class="accordion-button ${index > 0 ? 'collapsed' : ''}" type="button" data-bs-toggle="collapse" data-bs-target="#change${index}">
                                        Decoding Level ${change.level}
                                    </button>
                                </h2>
                                <div id="change${index}" class="accordion-collapse collapse ${index === 0 ? 'show' : ''}" data-bs-parent="#decodingChanges">
                                    <div class="accordion-body">
                                        <div class="row">
                                            <div class="col-md-6">
                                                <h6>Before</h6>
                                                <code class="d-block text-break">${change.original}</code>
                                            </div>
                                            <div class="col-md-6">
                                                <h6>After</h6>
                                                <code class="d-block text-break">${change.decoded}</code>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            `).join('')}
                        </div>
                    </div>
                    ` : ''}
                    <div class="mt-4">
                        <h5>Technical Analysis</h5>
                        <div class="row">
                            <div class="col-md-6">
                                <table class="table table-sm">
                                    <tbody>
                                        <tr>
                                            <td><strong>Percent Codes Found:</strong></td>
                                            <td>${(urlInput.match(/%[0-9A-Fa-f]{2}/g) || []).length}</td>
                                        </tr>
                                        <tr>
                                            <td><strong>Characters Decoded:</strong></td>
                                            <td>${urlInput.length - decodedResult.length}</td>
                                        </tr>
                                        <tr>
                                            <td><strong>Compression Ratio:</strong></td>
                                            <td>${((1 - (decodedResult.length / urlInput.length)) * 100).toFixed(1)}%</td>
                                        </tr>
                                    </tbody>
                                </table>
                            </div>
                            <div class="col-md-6">
                                <table class="table table-sm">
                                    <tbody>
                                        <tr>
                                            <td><strong>Character Encoding:</strong></td>
                                            <td>UTF-8</td>
                                        </tr>
                                        <tr>
                                            <td><strong>URL Components:</strong></td>
                                            <td>${decodedResult.includes('?') ? 'Query present' : 'No query'}</td>
                                        </tr>
                                        <tr>
                                            <td><strong>Validation:</strong></td>
                                            <td>Passed</td>
                                        </tr>
                                    </tbody>
                                </table>
                            </div>
                        </div>
                    </div>
                    <div class="mt-4">
                        <h5>Common Encodings Detected</h5>
                        <div class="list-group">
                            <div class="list-group-item d-flex justify-content-between align-items-center">
                                <div>
                                    <strong>Space character (%20)</strong>
                                    <br><small class="text-muted">Most common URL encoding</small>
                                </div>
                                <span class="badge bg-primary">${(urlInput.match(/%20/g) || []).length}</span>
                            </div>
                            <div class="list-group-item d-flex justify-content-between align-items-center">
                                <div>
                                    <strong>Plus sign (%2B)</strong>
                                    <br><small class="text-muted">Mathematical operator</small>
                                </div>
                                <span class="badge bg-secondary">${(urlInput.match(/%2B/g) || []).length}</span>
                            </div>
                            <div class="list-group-item d-flex justify-content-between align-items-center">
                                <div>
                                    <strong>Ampersand (%26)</strong>
                                    <br><small class="text-muted">Query parameter separator</small>
                                </div>
                                <span class="badge bg-info">${(urlInput.match(/%26/g) || []).length}</span>
                            </div>
                            <div class="list-group-item d-flex justify-content-between align-items-center">
                                <div>
                                    <strong>Equals sign (%3D)</strong>
                                    <br><small class="text-muted">Parameter assignment</small>
                                </div>
                                <span class="badge bg-warning">${(urlInput.match(/%3D/g) || []).length}</span>
                            </div>
                        </div>
                    </div>
                    <div class="mt-4">
                        <h5>Analysis & Recommendations</h5>
                        <ul class="list-group">
                            <li class="list-group-item list-group-item-success">✓ URL decoding completed successfully</li>
                            <li class="list-group-item list-group-item-success">✓ All percent-encoded characters properly decoded</li>
                            <li class="list-group-item list-group-item-info">ℹ ${decodeLevel === 'double' ? 'Double decoding applied for nested encoding' : 'Single decoding applied'}</li>
                            <li class="list-group-item list-group-item-info">ℹ Character encoding detected as UTF-8</li>
                            <li class="list-group-item list-group-item-success">✓ Output is valid and properly formatted</li>
                        </ul>
                    </div>
                    <div class="mt-4">
                        <h5>Quick Actions</h5>
                        <div class="d-flex gap-2 flex-wrap">
                            <button class="btn btn-primary" onclick="encodeToUrl()">
                                Re-encode to URL
                            </button>
                            <button class="btn btn-outline-primary" onclick="validateUrl()">
                                Validate URL
                            </button>
                            <button class="btn btn-outline-primary" onclick="extractParams()">
                                Extract Parameters
                            </button>
                            <button class="btn btn-outline-secondary" onclick="clearResults()">
                                Clear Results
                            </button>
                        </div>
                    </div>
                `;

                resultsDiv.innerHTML = mockResults;
                document.getElementById('resultsSection').style.display = 'block';
            }, 1500);
        });

        function copyResult() {
            const output = document.getElementById('decodedOutput').textContent || document.getElementById('decodedOutput').innerText;
            navigator.clipboard.writeText(output).then(() => {
                alert('Decoded URL copied to clipboard!');
            });
        }

        function encodeToUrl() {
            const decoded = document.getElementById('decodedOutput').textContent || document.getElementById('decodedOutput').innerText;
            const encoded = encodeURIComponent(decoded);
            document.getElementById('urlInput').value = encoded;
            alert('Result re-encoded to URL format and placed in input field.');
        }

        function validateUrl() {
            const decoded = document.getElementById('decodedOutput').textContent || document.getElementById('decodedOutput').innerText;
            try {
                new URL(decoded);
                alert('✓ URL is valid and properly formatted.');
            } catch (e) {
                alert('⚠ URL validation failed. The decoded string may not be a complete URL.');
            }
        }

        function extractParams() {
            const decoded = document.getElementById('decodedOutput').textContent || document.getElementById('decodedOutput').innerText;
            try {
                const url = new URL(decoded);
                const params = {};
                for (let [key, value] of url.searchParams) {
                    params[key] = value;
                }
                const paramString = Object.entries(params).map(([k, v]) => `${k}: ${v}`).join('\n');
                alert(`Query Parameters Extracted:\n\n${paramString || 'No query parameters found.'}`);
            } catch (e) {
                alert('Unable to extract parameters. The decoded string may not be a valid URL.');
            }
        }

        function clearResults() {
            document.getElementById('resultsSection').style.display = 'none';
            document.getElementById('urlInput').value = '';
        }
    </script>

<?php include __DIR__ . '/../../includes/footer.php'; ?>