<?php
$page_title = "Slug Generator - Loganix SEO Agency";
include_once __DIR__ . '/../../includes/config.php';
include_once __DIR__ . '/../../includes/header.php';
?>

    <!-- Tool Hero Section -->
    <section class="hero-section" style="background: linear-gradient(135deg, #0153C8 0%, #092D57 100%);">
        <div class="container">
            <div class="row align-items-center">
                <div class="col-12 text-center hero-content">
                    <h1 class="fade-in" style="font-size: 2.5rem;">Slug Generator</h1>
                    <p class="fade-in">Generate SEO-friendly URL slugs from titles and keywords for clean, readable web addresses.</p>
                </div>
            </div>
        </div>
    </section>

    <!-- Tool Interface -->
    <section class="py-5">
        <div class="container">
            <div class="row">
                <div class="col-lg-10 mx-auto">
                    <div class="service-card p-4">
                        <h3 class="mb-4">Generate URL Slug</h3>
                        <form id="slugGeneratorForm">
                            <div class="mb-3">
                                <label for="inputText" class="form-label">Title or Text</label>
                                <input type="text" class="form-control" id="inputText" placeholder="Enter your page title or keywords" required>
                            </div>
                            <div class="mb-3">
                                <label for="separatorType" class="form-label">Word Separator</label>
                                <select class="form-control" id="separatorType">
                                    <option value="hyphen">-</option>
                                    <option value="underscore">_</option>
                                    <option value="none">None (concatenated)</option>
                                </select>
                            </div>
                            <div class="mb-3">
                                <label for="maxLength" class="form-label">Maximum Length (Optional)</label>
                                <input type="number" class="form-control" id="maxLength" placeholder="Leave empty for no limit" min="10" max="100">
                            </div>
                            <div class="mb-3">
                                <div class="form-check">
                                    <input class="form-check-input" type="checkbox" id="removeStopWords" checked>
                                    <label class="form-check-label" for="removeStopWords">
                                        Remove common stop words (the, and, or, etc.)
                                    </label>
                                </div>
                            </div>
                            <div class="mb-3">
                                <div class="form-check">
                                    <input class="form-check-input" type="checkbox" id="lowercase" checked>
                                    <label class="form-check-label" for="lowercase">
                                        Convert to lowercase
                                    </label>
                                </div>
                            </div>
                            <button type="submit" class="btn btn-primary">Generate Slug</button>
                        </form>
                    </div>

                    <!-- Results Section -->
                    <div id="resultsSection" class="mt-4" style="display: none;">
                        <div class="service-card p-4">
                            <h4>Slug Generation Results</h4>
                            <div id="slugResults" class="mt-4"></div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </section>

    <!-- Article Section -->
    <section class="py-5 bg-light">
        <div class="container">
            <div class="row">
                <div class="col-lg-10 mx-auto">
                    <article>
                        <h2>The Complete Guide to URL Slug Generation for SEO and User Experience</h2>

                        <p>URL slug generation is essential for creating clean, readable, and SEO-friendly web addresses. A slug generator transforms page titles and keywords into URL-friendly formats that improve search engine visibility and user experience. This comprehensive guide explores slug creation best practices, SEO implications, and technical considerations.</p>

                        <h3>What is a URL Slug?</h3>

                        <p>A URL slug is the human-readable part of a web address that comes after the domain name. It's the descriptive text that identifies a specific page or resource in a clean, readable format.</p>

                        <h4>Slug Examples</h4>

                        <pre><code>Good slug: https://example.com/blog/seo-friendly-url-guide
Bad slug:  https://example.com/blog?id=123&category=seo&page=guide</code></pre>

                        <h3>Why URL Slugs Matter for SEO</h3>

                        <p>Well-crafted slugs contribute significantly to search engine optimization:</p>

                        <h4>Keyword Optimization</h4>

                        <p>Slugs provide keyword context to search engines:</p>

                        <ul>
                            <li>Include target keywords naturally</li>
                            <li>Help search engines understand page content</li>
                            <li>Improve relevance signals</li>
                        </ul>

                        <h4>User Experience</h4>

                        <p>Clean slugs enhance user interaction:</p>

                        <ul>
                            <li>Readable and memorable URLs</li>
                            <li>Indicate content before clicking</li>
                            <li>Shareable and bookmarkable</li>
                        </ul>

                        <h4>Click-Through Rates</h4>

                        <p>Descriptive slugs appear in search results:</p>

                        <ul>
                            <li>Keywords may appear bold in SERPs</li>
                            <li>Clear descriptions encourage clicks</li>
                            <li>Professional appearance builds trust</li>
                        </ul>

                        <h3>How Slug Generators Work</h3>

                        <p>Slug generation involves systematic text transformation:</p>

                        <ol>
                            <li><strong>Text Input:</strong> Accept title or keyword phrase</li>
                            <li><strong>Cleaning:</strong> Remove special characters and formatting</li>
                            <li><strong>Normalization:</strong> Convert to consistent case</li>
                            <li><strong>Word Separation:</strong> Apply chosen separator</li>
                            <li><strong>Optimization:</strong> Remove stop words and optimize length</li>
                            <li><strong>Validation:</strong> Ensure URL compatibility</li>
                        </ol>

                        <h3>Slug Generation Best Practices</h3>

                        <h4>Length Optimization</h4>

                        <p>Keep slugs concise yet descriptive:</p>

                        <ul>
                            <li>Ideal length: 50-60 characters</li>
                            <li>Include primary keyword</li>
                            <li>Avoid excessive words</li>
                        </ul>

                        <h4>Keyword Inclusion</h4>

                        <p>Incorporate relevant keywords strategically:</p>

                        <ul>
                            <li>Primary keyword near the beginning</li>
                            <li>Natural keyword integration</li>
                            <li>Avoid keyword stuffing</li>
                        </ul>

                        <h4>Readability</h4>

                        <p>Ensure slugs are human-readable:</p>

                        <ul>
                            <li>Use hyphens for word separation</li>
                            <li>Avoid abbreviations when possible</li>
                            <li>Maintain logical word order</li>
                        </ul>

                        <h3>Common Slug Mistakes to Avoid</h3>

                        <h4>Non-Descriptive Slugs</h4>

                        <p>Avoid generic or meaningless slugs:</p>

                        <pre><code>❌ /page?id=123
❌ /post/untitled
✅ /seo-guide-for-beginners</code></pre>

                        <h4>Over-Optimization</h4>

                        <p>Don't force too many keywords:</p>

                        <pre><code>❌ /best-seo-tools-free-online-seo-keyword-research-tools-guide
✅ /best-free-seo-keyword-research-tools</code></pre>

                        <h4>Special Characters</h4>

                        <p>Avoid characters that cause issues:</p>

                        <pre><code>❌ /seo-guide-&-tips
❌ /what's-new-in-seo
✅ /seo-guide-and-tips
✅ /whats-new-in-seo</code></pre>

                        <h3>Slug Types and Formats</h3>

                        <h4>Hyphenated Slugs</h4>

                        <p>Most common and SEO-friendly:</p>

                        <pre><code>seo-friendly-url-guide
best-seo-practices-2024</code></pre>

                        <h4>Underscore Slugs</h4>

                        <p>Less common but still valid:</p>

                        <pre><code>seo_friendly_url_guide
best_seo_practices_2024</code></pre>

                        <h4>No Separator Slugs</h4>

                        <p>Compact but less readable:</p>

                        <pre><code>seofriendlyurlguide
bestseopractices2024</code></pre>

                        <h3>Stop Words in Slugs</h3>

                        <h4>Common Stop Words</h4>

                        <p>Words often removed from slugs:</p>

                        <ul>
                            <li>Articles: a, an, the</li>
                            <li>Prepositions: in, on, at, for, with</li>
                            <li>Conjunctions: and, or, but</li>
                            <li>Pronouns: it, they, we</li>
                        </ul>

                        <h4>When to Keep Stop Words</h4>

                        <p>Sometimes stop words improve clarity:</p>

                        <ul>
                            <li>Essential for meaning: "to" in "how-to-guide"</li>
                            <li>Brand consistency</li>
                            <li>Common phrase preservation</li>
                        </ul>

                        <h3>Slug Optimization for Different Content Types</h3>

                        <h4>Blog Posts</h4>

                        <p>Descriptive and keyword-rich:</p>

                        <pre><code>/blog/complete-seo-guide-for-beginners
/blog/google-algorithm-updates-2024</code></pre>

                        <h4>Product Pages</h4>

                        <p>Include product and category:</p>

                        <pre><code>/products/seo-analysis-tool-pro
/tools/keyword-research-suite</code></pre>

                        <h4>Service Pages</h4>

                        <p>Focus on service benefits:</p>

                        <pre><code>/services/seo-consultation-services
/marketing/digital-marketing-agency</code></pre>

                        <h3>Technical Slug Considerations</h3>

                        <h4>URL Encoding</h4>

                        <p>Handle special characters properly:</p>

                        <ul>
                            <li>Spaces become %20 or hyphens</li>
                            <li>Unicode characters need encoding</li>
                            <li>Avoid problematic symbols</li>
                        </ul>

                        <h4>Case Sensitivity</h4>

                        <p>URL case handling varies:</p>

                        <ul>
                            <li>Most systems are case-insensitive</li>
                            <li>Convention favors lowercase</li>
                            <li>Consistency is key</li>
                        </ul>

                        <h3>Slug Migration and Redirects</h3>

                        <h4>Changing Slugs</h4>

                        <p>Handle slug changes carefully:</p>

                        <ul>
                            <li>Set up 301 redirects from old slugs</li>
                            <li>Update internal links</li>
                            <li>Monitor for crawl errors</li>
                        </ul>

                        <h4>Slug Versioning</h4>

                        <p>Handle content updates:</p>

                        <ul>
                            <li>Keep slugs stable when possible</li>
                            <li>Use dates for time-sensitive content</li>
                            <li>Maintain backward compatibility</li>
                        </ul>

                        <h3>Measuring Slug Performance</h3>

                        <p>Track slug effectiveness:</p>

                        <ul>
                            <li><strong>Search Rankings:</strong> Monitor keyword performance</li>
                            <li><strong>Click-Through Rates:</strong> SERP appearance impact</li>
                            <li><strong>User Engagement:</strong> Time on page and bounce rates</li>
                            <li><strong>Shareability:</strong> Social sharing and linking</li>
                        </ul>

                        <h3>Tools for Slug Generation</h3>

                        <p>Various tools help create optimized slugs:</p>

                        <ul>
                            <li><strong>CMS Plugins:</strong> Automatic slug generation in WordPress, etc.</li>
                            <li><strong>Online Generators:</strong> Web-based slug creation tools</li>
                            <li><strong>Programming Libraries:</strong> Slug generation functions</li>
                            <li><strong>SEO Tools:</strong> Integrated slug optimization</li>
                        </ul>

                        <h3>Advanced Slug Strategies</h3>

                        <h4>Dynamic Slug Generation</h4>

                        <p>Automated slug creation:</p>

                        <ul>
                            <li>Server-side slug generation</li>
                            <li>Real-time optimization</li>
                            <li>Multi-language support</li>
                        </ul>

                        <h4>Slug Personalization</h4>

                        <p>Custom slug creation:</p>

                        <ul>
                            <li>Brand-specific formatting</li>
                            <li>Category-based patterns</li>
                            <li>Content type indicators</li>
                        </ul>

                        <h3>Slug SEO Impact Analysis</h3>

                        <h4>Search Engine Processing</h4>

                        <p>How search engines handle slugs:</p>

                        <ul>
                            <li>Slugs provide content context</li>
                            <li>Keywords help ranking algorithms</li>
                            <li>Clean URLs improve crawlability</li>
                        </ul>

                        <h4>Duplicate Content Issues</h4>

                        <p>Avoid slug conflicts:</p>

                        <ul>
                            <li>Unique slugs for each page</li>
                            <li>Handle similar titles</li>
                            <li>Use canonical tags when needed</li>
                        </ul>

                        <h3>Future of URL Slugs</h3>

                        <p>URL slug technology continues to evolve:</p>

                        <ul>
                            <li><strong>AI-Powered Optimization:</strong> Machine learning for better slugs</li>
                            <li><strong>Voice Search Adaptation:</strong> Conversational slug formats</li>
                            <li><strong>International SEO:</strong> Multi-language slug handling</li>
                            <li><strong>Rich Snippets Integration:</strong> Structured data in slugs</li>
                        </ul>

                        <h3>Conclusion</h3>

                        <p>URL slug generation is a critical aspect of technical SEO and user experience. A slug generator helps create clean, descriptive, and SEO-friendly URLs that improve search engine visibility and user engagement. By following best practices and understanding slug optimization principles, you can enhance your website's search performance and professional appearance.</p>

                        <p>Remember that slugs should be descriptive, concise, and keyword-optimized while remaining user-friendly. Regular review and optimization of your URL structure can significantly impact your SEO success.</p>

                        <p>Combine slug generation with other SEO tools like our <a href="<?php echo $base_url; ?>/tools/seo-friendly-url-generator">SEO-friendly URL generator</a> and <a href="<?php echo $base_url; ?>/tools/meta-tag-extractor">meta tag extractor</a> for comprehensive URL optimization.</p>

                        <p>For more information on URL slugs and SEO, check the <a href="https://developers.google.com/search/docs/advanced/guidelines/url-structure" target="_blank" rel="noopener">Google URL structure guidelines</a> and <a href="https://moz.com/learn/seo/url" target="_blank" rel="noopener">Moz's URL guide</a>. Start generating optimized slugs today and improve your website's SEO performance.</p>
                    </article>
                </div>
            </div>
        </div>
    </section>

    <!-- Related Tools -->
    <section class="py-5">
        <div class="container">
            <div class="section-title fade-in">
                <h2>Related SEO Tools</h2>
                <p>Explore our other powerful SEO analysis tools</p>
            </div>
            <div class="row">
                <div class="col-lg-4 fade-in">
                    <div class="service-card">
                        <div class="icon">🔗</div>
                        <h4>SEO-Friendly URL Generator</h4>
                        <p>Generate clean, optimized URLs for better search engine rankings.</p>
                        <a href="<?php echo $base_url; ?>/tools/seo-friendly-url-generator" class="btn btn-primary mt-3">Use Tool</a>
                    </div>
                </div>
                <div class="col-lg-4 fade-in">
                    <div class="service-card">
                        <div class="icon">📝</div>
                        <h4>Text Case Converter</h4>
                        <p>Convert text between different case formats for consistent styling.</p>
                        <a href="<?php echo $base_url; ?>/tools/text-case-converter" class="btn btn-primary mt-3">Use Tool</a>
                    </div>
                </div>
                <div class="col-lg-4 fade-in">
                    <div class="service-card">
                        <div class="icon">🏷️</div>
                        <h4>Meta Tag Extractor</h4>
                        <p>Extract and analyze meta tags from any webpage for SEO optimization.</p>
                        <a href="<?php echo $base_url; ?>/tools/meta-tag-extractor" class="btn btn-primary mt-3">Use Tool</a>
                    </div>
                </div>
            </div>
        </div>
    </section>

    <!-- CTA Section -->
    <section class="cta-section">
        <div class="container text-center fade-in">
            <h2>Need Professional SEO Services?</h2>
            <p>While our tools are great for analysis, our expert team can help you implement advanced SEO strategies.</p>
            <a href="<?php echo $base_url; ?>/pages/contact" class="btn btn-light btn-lg">Get Expert Help</a>
        </div>
    </section>

    <script>
        // Common stop words to remove
        const stopWords = ['a', 'an', 'and', 'are', 'as', 'at', 'be', 'by', 'for', 'from', 'has', 'he', 'in', 'is', 'it', 'its', 'of', 'on', 'that', 'the', 'to', 'was', 'will', 'with'];

        document.getElementById('slugGeneratorForm').addEventListener('submit', function(e) {
            e.preventDefault();

            const inputText = document.getElementById('inputText').value.trim();
            const separatorType = document.getElementById('separatorType').value;
            const maxLength = document.getElementById('maxLength').value;
            const removeStopWords = document.getElementById('removeStopWords').checked;
            const lowercase = document.getElementById('lowercase').checked;
            const resultsDiv = document.getElementById('slugResults');

            if (!inputText) {
                alert('Please enter text to generate a slug.');
                return;
            }

            resultsDiv.innerHTML = '<div class="text-center"><div class="spinner-border" role="status"><span class="visually-hidden">Loading...</span></div><p class="mt-2">Generating URL slug...</p></div>';

            // In a real implementation, this would generate the slug
            // For demo purposes, we'll simulate the results
            setTimeout(() => {
                // Generate slug
                let slug = inputText;

                // Convert to lowercase if requested
                if (lowercase) {
                    slug = slug.toLowerCase();
                }

                // Remove special characters and replace with spaces
                slug = slug.replace(/[^\w\s-]/g, ' ');

                // Split into words
                let words = slug.split(/\s+/).filter(word => word.length > 0);

                // Remove stop words if requested
                if (removeStopWords) {
                    words = words.filter(word => !stopWords.includes(word.toLowerCase()));
                }

                // Apply separator
                let separator = '';
                switch(separatorType) {
                    case 'hyphen':
                        separator = '-';
                        break;
                    case 'underscore':
                        separator = '_';
                        break;
                    case 'none':
                        separator = '';
                        break;
                }

                slug = words.join(separator);

                // Apply max length if specified
                if (maxLength && slug.length > maxLength) {
                    slug = slug.substring(0, maxLength);
                    // Remove trailing separator
                    if (separator && slug.endsWith(separator)) {
                        slug = slug.slice(0, -1);
                    }
                }

                // Final cleanup - remove multiple separators
                if (separator) {
                    slug = slug.replace(new RegExp(`${separator}+`, 'g'), separator);
                    slug = slug.replace(new RegExp(`^${separator}|${separator}$`, 'g'), '');
                }

                const mockResults = `
                    <div class="alert alert-success">
                        <strong>✓ Slug Generated Successfully!</strong>
                    </div>
                    <div class="row mt-4">
                        <div class="col-md-6">
                            <h5>Generated Slug</h5>
                            <div class="input-group">
                                <input type="text" class="form-control" id="generatedSlug" value="${slug}" readonly>
                                <button class="btn btn-outline-primary" type="button" onclick="copySlug()">
                                    <i class="fas fa-copy"></i> Copy
                                </button>
                            </div>
                            <div class="mt-2">
                                <small class="text-muted">Full URL example: https://example.com/${slug}</small>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <h5>Slug Analysis</h5>
                            <div class="alert alert-info">
                                <strong>Length:</strong> ${slug.length} characters<br>
                                <strong>Words:</strong> ${words.length}<br>
                                <strong>Separator:</strong> ${separatorType === 'hyphen' ? 'Hyphen (-)' : separatorType === 'underscore' ? 'Underscore (_)' : 'None'}<br>
                                <strong>Stop Words Removed:</strong> ${removeStopWords ? 'Yes' : 'No'}
                            </div>
                        </div>
                    </div>
                    <div class="mt-4">
                        <h5>Before & After</h5>
                        <div class="row">
                            <div class="col-md-6">
                                <h6>Original Text</h6>
                                <div class="bg-light p-3 rounded">
                                    <small>${inputText}</small>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <h6>Generated Slug</h6>
                                <div class="bg-light p-3 rounded">
                                    <small>${slug}</small>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="mt-4">
                        <h5>SEO Analysis</h5>
                        <ul class="list-group">
                            <li class="list-group-item list-group-item-${slug.length <= 60 ? 'success' : 'warning'}">${slug.length <= 60 ? '✓' : '⚠'} Length ${slug.length <= 60 ? 'optimal' : 'could be shorter'} (${slug.length}/60 recommended)</li>
                            <li class="list-group-item list-group-item-success">✓ Lowercase formatting applied</li>
                            <li class="list-group-item list-group-item-success">✓ Special characters removed</li>
                            <li class="list-group-item list-group-item-success">✓ URL-safe characters only</li>
                            <li class="list-group-item list-group-item-info">ℹ Contains ${words.length} keyword${words.length !== 1 ? 's' : ''} for SEO</li>
                            <li class="list-group-item list-group-item-${separatorType === 'hyphen' ? 'success' : 'info'}">${separatorType === 'hyphen' ? '✓' : 'ℹ'} ${separatorType === 'hyphen' ? 'Hyphen separator is SEO-friendly' : separatorType === 'underscore' ? 'Underscore separator used' : 'No separator (compact format)'}</li>
                        </ul>
                    </div>
                    <div class="mt-4">
                        <h5>Alternative Slug Suggestions</h5>
                        <div class="list-group">
                            <a href="#" class="list-group-item list-group-item-action" onclick="selectAlternative('${generateAlternativeSlug(words, 'hyphen', false)}')">
                                <strong>With Stop Words:</strong> ${generateAlternativeSlug(words, 'hyphen', false)}
                            </a>
                            <a href="#" class="list-group-item list-group-item-action" onclick="selectAlternative('${generateAlternativeSlug(words, 'underscore', removeStopWords)}')">
                                <strong>Underscore Style:</strong> ${generateAlternativeSlug(words, 'underscore', removeStopWords)}
                            </a>
                            <a href="#" class="list-group-item list-group-item-action" onclick="selectAlternative('${generateAlternativeSlug(words.slice(0, 3), 'hyphen', removeStopWords)}')">
                                <strong>Short Version:</strong> ${generateAlternativeSlug(words.slice(0, 3), 'hyphen', removeStopWords)}
                            </a>
                            <a href="#" class="list-group-item list-group-item-action" onclick="selectAlternative('${generateAlternativeSlug(words, 'none', removeStopWords)}')">
                                <strong>Compact:</strong> ${generateAlternativeSlug(words, 'none', removeStopWords)}
                            </a>
                        </div>
                    </div>
                    <div class="mt-4">
                        <h5>Slug Performance Metrics</h5>
                        <div class="row text-center">
                            <div class="col-md-3">
                                <div class="card">
                                    <div class="card-body">
                                        <h6>Readability</h6>
                                        <h4 class="text-success">${calculateReadability(words)}/10</h4>
                                        <small>Score</small>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-3">
                                <div class="card">
                                    <div class="card-body">
                                        <h6>SEO Potential</h6>
                                        <h4 class="text-primary">${calculateSEOPotential(words)}/10</h4>
                                        <small>Score</small>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-3">
                                <div class="card">
                                    <div class="card-body">
                                        <h6>Character Efficiency</h6>
                                        <h4 class="text-info">${Math.round((slug.length / inputText.length) * 100)}%</h4>
                                        <small>Of original</small>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-3">
                                <div class="card">
                                    <div class="card-body">
                                        <h6>Words Retained</h6>
                                        <h4 class="text-warning">${words.length}</h4>
                                        <small>Keywords</small>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="mt-4">
                        <h5>Usage Examples</h5>
                        <div class="bg-light p-3 rounded">
                            <small>
                                <strong>Blog Post:</strong> https://yoursite.com/blog/${slug}<br>
                                <strong>Product Page:</strong> https://yoursite.com/products/${slug}<br>
                                <strong>Service Page:</strong> https://yoursite.com/services/${slug}<br>
                                <strong>Category:</strong> https://yoursite.com/category/${slug}
                            </small>
                        </div>
                    </div>
                    <div class="mt-4">
                        <h5>Quick Actions</h5>
                        <div class="d-flex gap-2 flex-wrap">
                            <button class="btn btn-primary" onclick="downloadSlug()">
                                Download Slug
                            </button>
                            <button class="btn btn-outline-primary" onclick="generatePermalink()">
                                Generate Permalink
                            </button>
                            <button class="btn btn-outline-primary" onclick="validateSlug()">
                                Validate Slug
                            </button>
                            <button class="btn btn-outline-secondary" onclick="shareSlug()">
                                Share Slug
                            </button>
                        </div>
                    </div>
                `;

                resultsDiv.innerHTML = mockResults;
                document.getElementById('resultsSection').style.display = 'block';
            }, 1000);
        });

        function generateAlternativeSlug(words, separator, removeStops) {
            let filteredWords = words;
            if (removeStops) {
                filteredWords = words.filter(word => !stopWords.includes(word.toLowerCase()));
            }

            switch(separator) {
                case 'hyphen':
                    return filteredWords.join('-');
                case 'underscore':
                    return filteredWords.join('_');
                case 'none':
                    return filteredWords.join('');
                default:
                    return filteredWords.join('-');
            }
        }

        function calculateReadability(words) {
            // Simple readability calculation
            const avgWordLength = words.reduce((sum, word) => sum + word.length, 0) / words.length;
            const score = Math.max(1, Math.min(10, 10 - (avgWordLength - 5) * 0.5));
            return Math.round(score);
        }

        function calculateSEOPotential(words) {
            // Simple SEO potential calculation
            const longWords = words.filter(word => word.length > 6).length;
            const score = Math.min(10, words.length + longWords);
            return Math.round(score);
        }

        function copySlug() {
            const slugInput = document.getElementById('generatedSlug');
            slugInput.select();
            document.execCommand('copy');

            // Show feedback
            const button = event.target;
            const originalText = button.innerHTML;
            button.innerHTML = '<i class="fas fa-check"></i> Copied!';
            button.classList.remove('btn-outline-primary');
            button.classList.add('btn-success');

            setTimeout(() => {
                button.innerHTML = originalText;
                button.classList.remove('btn-success');
                button.classList.add('btn-outline-primary');
            }, 2000);
        }

        function selectAlternative(slug) {
            document.getElementById('generatedSlug').value = slug;
        }

        function downloadSlug() {
            const slug = document.getElementById('generatedSlug').value;
            const blob = new Blob([slug], { type: 'text/plain' });
            const url = URL.createObjectURL(blob);
            const a = document.createElement('a');
            a.href = url;
            a.download = 'url-slug.txt';
            document.body.appendChild(a);
            a.click();
            document.body.removeChild(a);
            URL.revokeObjectURL(url);
        }

        function generatePermalink() {
            const slug = document.getElementById('generatedSlug').value;
            const permalink = `https://example.com/${slug}`;
            navigator.clipboard.writeText(permalink).then(() => {
                alert('Permalink copied to clipboard: ' + permalink);
            });
        }

        function validateSlug() {
            const slug = document.getElementById('generatedSlug').value;
            const issues = [];

            if (slug.length > 100) issues.push('Slug is very long (>100 characters)');
            if (slug.includes(' ')) issues.push('Contains spaces');
            if (/[^a-zA-Z0-9\-_]/.test(slug)) issues.push('Contains special characters');
            if (slug.startsWith('-') || slug.endsWith('-')) issues.push('Starts or ends with separator');

            if (issues.length === 0) {
                alert('✓ Slug is valid and SEO-friendly!');
            } else {
                alert('Issues found:\n' + issues.join('\n'));
            }
        }

        function shareSlug() {
            const slug = document.getElementById('generatedSlug').value;
            if (navigator.share) {
                navigator.share({
                    title: 'Generated URL Slug',
                    text: `Check out this SEO-friendly slug: ${slug}`,
                    url: window.location.href
                });
            } else {
                navigator.clipboard.writeText(slug).then(() => {
                    alert('Slug copied to clipboard for sharing!');
                });
            }
        }
    </script>

<?php include __DIR__ . '/../../includes/footer.php'; ?>