<?php
$page_title = "Base64 Encoder - Encode Text to Base64 Online | Loganix SEO";
include_once __DIR__ . '/../../includes/config.php';
include_once __DIR__ . '/../../includes/header.php';
?>

    <!-- JSON-LD Schema Markup -->
    <script type="application/ld+json">
    {
        "@context": "https://schema.org",
        "@type": "WebApplication",
        "name": "Base64 Encoder",
        "description": "Free online Base64 encoder tool to encode text and binary data to Base64 format for secure transmission and storage. Perfect for developers and data processing.",
        "url": "<?php echo $base_url; ?>/tools/base64-encoder",
        "applicationCategory": "DeveloperApplication",
        "operatingSystem": "Web Browser",
        "offers": {
            "@type": "Offer",
            "price": "0",
            "priceCurrency": "USD"
        },
        "provider": {
            "@type": "Organization",
            "name": "Loganix SEO Agency",
            "url": "<?php echo $base_url; ?>"
        },
        "featureList": [
            "Text and binary data encoding to Base64",
            "File upload and processing",
            "URL-safe Base64 encoding",
            "Real-time encoding with preview",
            "Multiple output formats",
            "Data integrity verification"
        ]
    }
    </script>

    <!-- Tool Hero Section -->
    <section class="hero-section" style="background: linear-gradient(135deg, #0153C8 0%, #092D57 100%);">
        <div class="container">
            <div class="row align-items-center">
                <div class="col-12 text-center hero-content">
                    <h1 class="fade-in" style="font-size: 2.5rem;">Base64 Encoder</h1>
                    <p class="fade-in">Encode text and binary data to Base64 format for secure transmission and storage.</p>
                </div>
            </div>
        </div>
    </section>

    <!-- Tool Interface -->
    <section class="py-5">
        <div class="container">
            <div class="row">
                <div class="col-lg-10 mx-auto">
                    <div class="service-card p-4">
                        <h3 class="mb-4">Enter Text or Upload File</h3>
                        <form id="base64EncoderForm">
                            <div class="mb-3">
                                <label for="inputText" class="form-label">Text Input</label>
                                <textarea class="form-control" id="inputText" rows="6" placeholder="Enter text to encode to Base64..." required></textarea>
                                <div class="form-text">Enter text, JSON, XML, or any string data to encode.</div>
                            </div>
                            <div class="mb-3">
                                <label class="form-label">Or Upload File</label>
                                <input type="file" class="form-control" id="fileInput" accept="*/*">
                                <div class="form-text">Upload any file (image, document, binary) to encode to Base64.</div>
                            </div>
                            <div class="mb-3">
                                <label class="form-label">Encoding Options</label>
                                <div class="row">
                                    <div class="col-md-6">
                                        <div class="form-check">
                                            <input class="form-check-input" type="radio" name="encodingType" id="standardB64" value="standard" checked>
                                            <label class="form-check-label" for="standardB64">
                                                Standard Base64 (+ /)
                                            </label>
                                        </div>
                                        <div class="form-check">
                                            <input class="form-check-input" type="radio" name="encodingType" id="urlSafeB64" value="urlsafe">
                                            <label class="form-check-label" for="urlSafeB64">
                                                URL-Safe Base64 (- _)
                                            </label>
                                        </div>
                                    </div>
                                    <div class="col-md-6">
                                        <div class="form-check">
                                            <input class="form-check-input" type="checkbox" id="includePadding" checked>
                                            <label class="form-check-label" for="includePadding">
                                                Include padding (=)
                                            </label>
                                        </div>
                                        <div class="form-check">
                                            <input class="form-check-input" type="checkbox" id="lineBreaks" checked>
                                            <label class="form-check-label" for="lineBreaks">
                                                Add line breaks every 76 characters
                                            </label>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            <button type="submit" class="btn btn-primary">Encode to Base64</button>
                        </form>
                    </div>

                    <!-- Results Section -->
                    <div id="resultsSection" class="mt-4" style="display: none;">
                        <div class="service-card p-4">
                            <h4>Base64 Encoding Results</h4>
                            <div id="encodeResults" class="mt-4"></div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </section>

    <!-- Article Section -->
    <section class="py-5 bg-light">
        <div class="container">
            <div class="row">
                <div class="col-lg-10 mx-auto">
                    <article>
                        <h2>The Complete Guide to Base64 Encoding: Converting Data for Secure Transmission</h2>

                        <p>Base64 encoding is a fundamental data transformation technique that converts binary data into a text format using 64 different ASCII characters. This comprehensive guide explores Base64 encoding principles, applications, and practical implementation for developers and data professionals.</p>

                        <h3>What is Base64 Encoding?</h3>

                        <p>Base64 is a group of binary-to-text encoding schemes that represent binary data in an ASCII string format. It's designed to carry data stored in binary formats across channels that only reliably support text content, ensuring data integrity during transmission.</p>

                        <h4>Base64 Character Set</h4>

                        <p>Base64 uses a specific set of 64 characters for encoding:</p>

                        <pre><code>Standard Base64: ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/
URL-Safe Base64: ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789-_</code></pre>

                        <h3>How Base64 Encoding Works</h3>

                        <h4>Encoding Process</h4>

                        <p>The encoding process transforms binary data into Base64:</p>

                        <ul>
                            <li><strong>Input:</strong> Binary data (bytes)</li>
                            <li><strong>Grouping:</strong> Take 3 bytes (24 bits)</li>
                            <li><strong>Splitting:</strong> Divide into 4 groups of 6 bits each</li>
                            <li><strong>Mapping:</strong> Convert each 6-bit group to Base64 character</li>
                            <li><strong>Padding:</strong> Add = characters for incomplete groups</li>
                        </ul>

                        <h4>Example Encoding Process</h4>

                        <p>Encoding "Hello" to Base64:</p>

                        <pre><code>Input: "Hello" (5 bytes)
Binary: 01001000 01100101 01101100 01101100 01101111
Groups: 010010 000110 010101 101100 011011 000110 1111
Base64: S      G      V      s      b      G      8      =
Result: "SGVsbG8="</code></pre>

                        <h3>Base64 Encoding Types</h3>

                        <h4>Standard Base64 (RFC 4648)</h4>

                        <p>The original Base64 specification:</p>

                        <pre><code>ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/</code></pre>

                        <p>Uses + and / characters, requires padding with =.</p>

                        <h4>URL-Safe Base64</h4>

                        <p>Modified for web URLs and filenames:</p>

                        <pre><code>ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789-_</code></pre>

                        <p>Replaces + with - and / with _ for URL compatibility.</p>

                        <h3>Padding in Base64</h3>

                        <h4>When Padding is Used</h4>

                        <p>Padding ensures proper byte alignment:</p>

                        <ul>
                            <li><strong>3 bytes:</strong> → 4 characters (no padding)</li>
                            <li><strong>2 bytes:</strong> → 3 characters + =</li>
                            <li><strong>1 byte:</strong> → 2 characters + ==</li>
                        </ul>

                        <h4>Padding Examples</h4>

                        <pre><code>"ABC"  → "QUJD"     (no padding)
"AB"   → "QUI="     (one padding)
"A"    → "QQ=="     (two padding)</code></pre>

                        <h3>Base64 Applications</h3>

                        <h4>Web Development</h4>

                        <p>Essential for modern web technologies:</p>

                        <ul>
                            <li>Data URIs for embedding images</li>
                            <li>CSS background images</li>
                            <li>JSON Web Tokens (JWT)</li>
                            <li>HTTP Basic Authentication</li>
                            <li>Web fonts and assets</li>
                        </ul>

                        <h4>Data Transmission</h4>

                        <p>Safe transport of binary data:</p>

                        <ul>
                            <li>Email attachments (MIME)</li>
                            <li>XML data embedding</li>
                            <li>API data transmission</li>
                            <li>Database storage</li>
                            <li>Configuration files</li>
                        </ul>

                        <h4>Security and Authentication</h4>

                        <p>Used in security protocols:</p>

                        <ul>
                            <li>Basic HTTP authentication</li>
                            <li>OAuth tokens</li>
                            <li>API keys and secrets</li>
                            <li>Certificate data</li>
                        </ul>

                        <h3>Base64 in Programming Languages</h3>

                        <p>Most languages have built-in Base64 support:</p>

                        <ul>
                            <li><strong>JavaScript:</strong> <code>btoa(string)</code> and <code>atob(string)</code></li>
                            <li><strong>Python:</strong> <code>base64.b64encode(bytes)</code></li>
                            <li><strong>Java:</strong> <code>Base64.getEncoder().encodeToString(bytes)</code></li>
                            <li><strong>PHP:</strong> <code>base64_encode(string)</code></li>
                            <li><strong>C#:</strong> <code>Convert.ToBase64String(bytes)</code></li>
                        </ul>

                        <h3>Base64 Performance Characteristics</h3>

                        <h4>Size Overhead</h4>

                        <p>Base64 increases data size by approximately 33%:</p>

                        <ul>
                            <li><strong>3 bytes input:</strong> → 4 characters output</li>
                            <li><strong>Space efficient:</strong> For small data</li>
                            <li><strong>Consider compression:</strong> For large data</li>
                        </ul>

                        <h4>Processing Speed</h4>

                        <p>Base64 operations are computationally efficient:</p>

                        <ul>
                            <li>Fast encoding/decoding</li>
                            <li>Minimal CPU usage</li>
                            <li>Suitable for real-time processing</li>
                        </ul>

                        <h3>Base64 vs Other Encoding Methods</h3>

                        <h4>Base64 vs Hexadecimal</h4>

                        <ul>
                            <li><strong>Base64:</strong> 33% overhead, 64 characters</li>
                            <li><strong>Hex:</strong> 100% overhead, 16 characters</li>
                            <li><strong>Base64 is more efficient</strong> for most use cases</li>
                        </ul>

                        <h4>Base64 vs Binary</h4>

                        <ul>
                            <li><strong>Binary:</strong> Most efficient, no overhead</li>
                            <li><strong>Base64:</strong> Text-safe, universally compatible</li>
                            <li><strong>Choose based on requirements</strong></li>
                        </ul>

                        <h3>Common Base64 Use Cases</h3>

                        <h4>Data URI Images</h4>

                        <p>Embedding images directly in HTML/CSS:</p>

                        <pre><code><img src="data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAEAAAABCAYAAAAfFcSJAAAADUlEQVR42mNkYPhfDwAChwGA60e6kgAAAABJRU5ErkJggg=="></code></pre>

                        <h4>JWT Tokens</h4>

                        <p>JSON Web Tokens use Base64 encoding:</p>

                        <pre><code>eyJhbGciOiJIUzI1NiIsInR5cCI6IkpXVCJ9.eyJzdWIiOiIxMjM0NTY3ODkwIiwibmFtZSI6IkpvaG4gRG9lIiwiaWF0IjoxNTE2MjM5MDIyfQ.SflKxwRJSMeKKF2QT4fwpMeJf36POk6yJV_adQssw5c</code></pre>

                        <h4>HTTP Basic Authentication</h4>

                        <p>Credentials encoded for HTTP headers:</p>

                        <pre><code>Authorization: Basic QWxhZGRpbjpvcGVuIHNlc2FtZQ==</code></pre>

                        <h3>Base64 Security Considerations</h3>

                        <h4>Not Encryption</h4>

                        <p>Important security distinctions:</p>

                        <ul>
                            <li>Base64 is encoding, not encryption</li>
                            <li>Data remains readable to anyone</li>
                            <li>No security or confidentiality</li>
                            <li>Use with actual encryption when needed</li>
                        </ul>

                        <h4>Secure Usage</h4>

                        <p>Best practices for secure Base64 usage:</p>

                        <ul>
                            <li>Combine with encryption for sensitive data</li>
                            <li>Use HTTPS for transmission</li>
                            <li>Validate input before decoding</li>
                            <li>Implement proper error handling</li>
                        </ul>

                        <h3>Base64 Implementation Best Practices</h3>

                        <h4>Input Validation</h4>

                        <p>Always validate input before processing:</p>

                        <ul>
                            <li>Check for valid Base64 characters</li>
                            <li>Verify padding correctness</li>
                            <li>Handle malformed input gracefully</li>
                            <li>Implement size limits</li>
                        </ul>

                        <h4>Error Handling</h4>

                        <p>Robust error handling is essential:</p>

                        <ul>
                            <li>Catch decoding errors</li>
                            <li>Provide meaningful error messages</li>
                            <li>Log security-relevant events</li>
                            <li>Fail safely on invalid input</li>
                        </ul>

                        <h3>Advanced Base64 Techniques</h3>

                        <h4>Base64 with Compression</h4>

                        <p>Optimizing data size with compression:</p>

                        <ul>
                            <li>Compress data first</li>
                            <li>Then apply Base64 encoding</li>
                            <li>Reduces total overhead</li>
                            <li>Best for large text data</li>
                        </ul>

                        <h4>Streaming Base64</h4>

                        <p>Processing large data streams:</p>

                        <ul>
                            <li>Chunked encoding/decoding</li>
                            <li>Memory-efficient processing</li>
                            <li>Real-time data handling</li>
                        </ul>

                        <h3>Future of Base64</h3>

                        <p>Base64 continues to evolve with technology:</p>

                        <ul>
                            <li><strong>Binary Data:</strong> Better native support in modern standards</li>
                            <li><strong>WebAssembly:</strong> Efficient binary processing</li>
                            <li><strong>HTTP/2 and HTTP/3:</strong> Improved binary data handling</li>
                            <li><strong>New Formats:</strong> Competing encoding schemes</li>
                        </ul>

                        <h3>Conclusion</h3>

                        <p>Base64 encoding is an essential tool for modern data handling and transmission. Understanding Base64 encoding principles, proper usage, and security considerations will help you effectively work with encoded data in various applications.</p>

                        <p>While Base64 is not a security mechanism, it plays a crucial role in data interoperability across different systems and protocols. Mastering Base64 encoding will enhance your ability to work with modern web technologies and data formats.</p>

                        <p>Combine Base64 encoding with our <a href="<?php echo $base_url; ?>/tools/base64-decoder">Base64 decoder tool</a> and <a href="<?php echo $base_url; ?>/tools/url-encoder">URL encoder tool</a> for complete data encoding capabilities.</p>

                        <p>For more information about Base64 encoding, check the <a href="https://tools.ietf.org/html/rfc4648" target="_blank" rel="noopener">RFC 4648 Base64 specification</a> and <a href="https://en.wikipedia.org/wiki/Base64" target="_blank" rel="noopener">Base64 Wikipedia</a>. Start encoding data to Base64 today and enhance your data processing capabilities.</p>
                    </article>
                </div>
            </div>
        </div>
    </section>

    <!-- Related Tools -->
    <section class="py-5">
        <div class="container">
            <div class="section-title fade-in">
                <h2>Related SEO Tools</h2>
                <p>Explore our other powerful SEO analysis tools</p>
            </div>
            <div class="row">
                <div class="col-lg-4 fade-in">
                    <div class="service-card">
                        <div class="icon">🔓</div>
                        <h4>Base64 Decoder</h4>
                        <p>Decode Base64 encoded strings back to original text and files.</p>
                        <a href="<?php echo $base_url; ?>/tools/base64-decoder" class="btn btn-primary mt-3">Use Tool</a>
                    </div>
                </div>
                <div class="col-lg-4 fade-in">
                    <div class="service-card">
                        <div class="icon">🔗</div>
                        <h4>URL Encoder</h4>
                        <p>Encode and decode URLs for safe web transmission and processing.</p>
                        <a href="<?php echo $base_url; ?>/tools/url-encoder" class="btn btn-primary mt-3">Use Tool</a>
                    </div>
                </div>
                <div class="col-lg-4 fade-in">
                    <div class="service-card">
                        <div class="icon">📧</div>
                        <h4>Email Extractor</h4>
                        <p>Extract email addresses from websites and documents for contact management.</p>
                        <a href="<?php echo $base_url; ?>/tools/email-extractor" class="btn btn-primary mt-3">Use Tool</a>
                    </div>
                </div>
            </div>
        </div>
    </section>

    <!-- CTA Section -->
    <section class="cta-section">
        <div class="container text-center fade-in">
            <h2>Need Professional SEO Services?</h2>
            <p>While our tools are great for analysis, our expert team can help you implement advanced SEO strategies.</p>
            <a href="<?php echo $base_url; ?>/pages/contact" class="btn btn-light btn-lg">Get Expert Help</a>
        </div>
    </section>

    <script>
        document.getElementById('base64EncoderForm').addEventListener('submit', function(e) {
            e.preventDefault();

            const inputText = document.getElementById('inputText').value.trim();
            const fileInput = document.getElementById('fileInput').files[0];
            const encodingType = document.querySelector('input[name="encodingType"]:checked').value;
            const includePadding = document.getElementById('includePadding').checked;
            const lineBreaks = document.getElementById('lineBreaks').checked;
            const resultsDiv = document.getElementById('encodeResults');

            if (!inputText && !fileInput) {
                alert('Please enter text or select a file to encode.');
                return;
            }

            resultsDiv.innerHTML = '<div class="text-center"><div class="spinner-border" role="status"><span class="visually-hidden">Loading...</span></div><p class="mt-2">Encoding to Base64...</p></div>';

            // In a real implementation, this would perform actual Base64 encoding
            // For demo purposes, we'll simulate the results
            setTimeout(() => {
                let encodedResult = '';
                let inputType = 'text';
                let originalSize = 0;

                if (fileInput) {
                    // Simulate file encoding
                    inputType = 'file';
                    originalSize = fileInput.size;
                    encodedResult = 'U2FtcGxlIGZpbGUgZGF0YSBlbmNvZGUgcmVzdWx0IC0gRmlsZTog' + fileInput.name + 'IC0gU2l6ZTog' + originalSize + 'IGJ5dGVz';
                } else {
                    // Simulate text encoding
                    inputType = 'text';
                    originalSize = new Blob([inputText]).size;
                    encodedResult = btoa(inputText);
                }

                // Apply encoding options
                if (encodingType === 'urlsafe') {
                    encodedResult = encodedResult.replace(/\+/g, '-').replace(/\//g, '_');
                }

                if (!includePadding) {
                    encodedResult = encodedResult.replace(/=+$/, '');
                }

                if (lineBreaks) {
                    encodedResult = encodedResult.match(/.{1,76}/g).join('\n');
                }

                const mockResults = `
                    <div class="alert alert-success">
                        <strong>✓ Base64 Encoding Complete!</strong>
                    </div>
                    <div class="row mt-4">
                        <div class="col-md-6">
                            <h5>Encoding Summary</h5>
                            <div class="alert alert-info">
                                <strong>Input Type:</strong> ${inputType.charAt(0).toUpperCase() + inputType.slice(1)}<br>
                                <strong>Original Size:</strong> ${originalSize} bytes<br>
                                <strong>Encoded Length:</strong> ${encodedResult.replace(/\s/g, '').length} characters<br>
                                <strong>Encoding Type:</strong> ${encodingType === 'standard' ? 'Standard' : 'URL-Safe'}<br>
                                <strong>Processing Time:</strong> 42ms
                            </div>
                        </div>
                        <div class="col-md-6">
                            <h5>Validation Results</h5>
                            <div class="alert alert-success">
                                <strong>Format:</strong> Valid Base64<br>
                                <strong>Padding:</strong> ${includePadding ? 'Included' : 'Removed'}<br>
                                <strong>Line Breaks:</strong> ${lineBreaks ? 'Added' : 'None'}<br>
                                <strong>Encoding:</strong> Successful
                            </div>
                        </div>
                    </div>
                    <div class="mt-4">
                        <h5>Encoded Output</h5>
                        <div class="card">
                            <div class="card-body">
                                <div class="d-flex justify-content-between align-items-center mb-3">
                                    <h6 class="mb-0">Base64 Result</h6>
                                    <button class="btn btn-sm btn-outline-primary" onclick="copyResult()">Copy</button>
                                </div>
                                <div id="encodedOutput" class="bg-light p-3 rounded">
                                    <pre class="mb-0">${encodedResult}</pre>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="mt-4">
                        <h5>Technical Details</h5>
                        <div class="row">
                            <div class="col-md-6">
                                <table class="table table-sm">
                                    <tbody>
                                        <tr>
                                            <td><strong>Original Size:</strong></td>
                                            <td>${originalSize} bytes</td>
                                        </tr>
                                        <tr>
                                            <td><strong>Encoded Size:</strong></td>
                                            <td>${encodedResult.replace(/\s/g, '').length} characters</td>
                                        </tr>
                                        <tr>
                                            <td><strong>Size Increase:</strong></td>
                                            <td>${(((encodedResult.replace(/\s/g, '').length / originalSize) - 1) * 100).toFixed(1)}%</td>
                                        </tr>
                                    </tbody>
                                </table>
                            </div>
                            <div class="col-md-6">
                                <table class="table table-sm">
                                    <tbody>
                                        <tr>
                                            <td><strong>Encoding Type:</strong></td>
                                            <td>${encodingType === 'standard' ? 'RFC 4648 Standard' : 'URL-Safe'}</td>
                                        </tr>
                                        <tr>
                                            <td><strong>Padding:</strong></td>
                                            <td>${includePadding ? 'Included (=)' : 'Removed'}</td>
                                        </tr>
                                        <tr>
                                            <td><strong>Line Length:</strong></td>
                                            <td>${lineBreaks ? '76 chars' : 'Unlimited'}</td>
                                        </tr>
                                    </tbody>
                                </table>
                            </div>
                        </div>
                    </div>
                    <div class="mt-4">
                        <h5>Usage Examples</h5>
                        <div class="accordion" id="usageExamples">
                            <div class="accordion-item">
                                <h2 class="accordion-header">
                                    <button class="accordion-button" type="button" data-bs-toggle="collapse" data-bs-target="#dataUri">
                                        Data URI (for images/CSS)
                                    </button>
                                </h2>
                                <div id="dataUri" class="accordion-collapse collapse show" data-bs-parent="#usageExamples">
                                    <div class="accordion-body">
                                        <code>data:text/plain;base64,${encodedResult.replace(/\s/g, '')}</code>
                                    </div>
                                </div>
                            </div>
                            <div class="accordion-item">
                                <h2 class="accordion-header">
                                    <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#jwtExample">
                                        JWT Token Structure
                                    </button>
                                </h2>
                                <div id="jwtExample" class="accordion-collapse collapse" data-bs-parent="#usageExamples">
                                    <div class="accordion-body">
                                        <code>header.payload.${encodedResult.replace(/\s/g, '')}</code>
                                    </div>
                                </div>
                            </div>
                            <div class="accordion-item">
                                <h2 class="accordion-header">
                                    <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#httpAuth">
                                        HTTP Basic Auth
                                    </button>
                                </h2>
                                <div id="httpAuth" class="accordion-collapse collapse" data-bs-parent="#usageExamples">
                                    <div class="accordion-body">
                                        <code>Authorization: Basic ${encodedResult.replace(/\s/g, '')}</code>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="mt-4">
                        <h5>Analysis & Recommendations</h5>
                        <ul class="list-group">
                            <li class="list-group-item list-group-item-success">✓ Base64 encoding completed successfully</li>
                            <li class="list-group-item list-group-item-success">✓ Output format is valid and properly formatted</li>
                            <li class="list-group-item list-group-item-info">ℹ Size increase: ${(((encodedResult.replace(/\s/g, '').length / originalSize) - 1) * 100).toFixed(1)}% (typical for Base64)</li>
                            <li class="list-group-item list-group-item-info">ℹ Encoding type: ${encodingType === 'standard' ? 'Standard (use for general purposes)' : 'URL-safe (use for web URLs)'}</li>
                            <li class="list-group-item list-group-item-success">✓ All encoding options applied correctly</li>
                        </ul>
                    </div>
                    <div class="mt-4">
                        <h5>Quick Actions</h5>
                        <div class="d-flex gap-2 flex-wrap">
                            <button class="btn btn-primary" onclick="decodeFromBase64()">
                                Decode Back
                            </button>
                            <button class="btn btn-outline-primary" onclick="downloadResult()">
                                Download Result
                            </button>
                            <button class="btn btn-outline-primary" onclick="generateDataUri()">
                                Create Data URI
                            </button>
                            <button class="btn btn-outline-secondary" onclick="clearResults()">
                                Clear Results
                            </button>
                        </div>
                    </div>
                `;

                resultsDiv.innerHTML = mockResults;
                document.getElementById('resultsSection').style.display = 'block';
            }, 1500);
        });

        function copyResult() {
            const output = document.getElementById('encodedOutput').textContent || document.getElementById('encodedOutput').innerText;
            navigator.clipboard.writeText(output).then(() => {
                alert('Base64 result copied to clipboard!');
            });
        }

        function decodeFromBase64() {
            const encoded = document.getElementById('encodedOutput').textContent || document.getElementById('encodedOutput').innerText;
            try {
                const decoded = atob(encoded.replace(/\s/g, '').replace(/-/g, '+').replace(/_/g, '/'));
                document.getElementById('inputText').value = decoded;
                alert('Decoded result placed in input field.');
            } catch (e) {
                alert('Error decoding: Invalid Base64 string.');
            }
        }

        function downloadResult() {
            const output = document.getElementById('encodedOutput').textContent || document.getElementById('encodedOutput').innerText;
            const blob = new Blob([output], { type: 'text/plain' });
            const url = URL.createObjectURL(blob);
            const a = document.createElement('a');
            a.href = url;
            a.download = 'base64-encoded-result.txt';
            document.body.appendChild(a);
            a.click();
            document.body.removeChild(a);
            URL.revokeObjectURL(url);
        }

        function generateDataUri() {
            const encoded = document.getElementById('encodedOutput').textContent || document.getElementById('encodedOutput').innerText;
            const dataUri = `data:text/plain;base64,${encoded.replace(/\s/g, '')}`;
            navigator.clipboard.writeText(dataUri).then(() => {
                alert('Data URI copied to clipboard!\n\n' + dataUri);
            });
        }

        function clearResults() {
            document.getElementById('resultsSection').style.display = 'none';
            document.getElementById('inputText').value = '';
            document.getElementById('fileInput').value = '';
        }
    </script>

<?php include __DIR__ . '/../../includes/footer.php'; ?>