<?php
$page_title = "Base64 Decoder - Decode Base64 to Text Online | Loganix SEO";
include_once __DIR__ . '/../../includes/config.php';
include_once __DIR__ . '/../../includes/header.php';
?>

    <!-- JSON-LD Schema Markup -->
    <script type="application/ld+json">
    {
        "@context": "https://schema.org",
        "@type": "WebApplication",
        "name": "Base64 Decoder",
        "description": "Free online Base64 decoder tool to decode Base64 encoded strings back to original text, images, and files. Perfect for developers and data processing.",
        "url": "<?php echo $base_url; ?>/tools/base64-decoder",
        "applicationCategory": "DeveloperApplication",
        "operatingSystem": "Web Browser",
        "offers": {
            "@type": "Offer",
            "price": "0",
            "priceCurrency": "USD"
        },
        "provider": {
            "@type": "Organization",
            "name": "Loganix SEO Agency",
            "url": "<?php echo $base_url; ?>"
        },
        "featureList": [
            "Base64 string decoding",
            "Text and binary data support",
            "File upload and processing",
            "Image decoding capabilities",
            "URL-safe Base64 decoding",
            "Real-time decoding"
        ]
    }
    </script>

    <!-- Tool Hero Section -->
    <section class="hero-section" style="background: linear-gradient(135deg, #0153C8 0%, #092D57 100%);">
        <div class="container">
            <div class="row align-items-center">
                <div class="col-12 text-center hero-content">
                    <h1 class="fade-in" style="font-size: 2.5rem;">Base64 Decoder</h1>
                    <p class="fade-in">Decode Base64 encoded strings back to original text, images, and files instantly.</p>
                </div>
            </div>
        </div>
    </section>

    <!-- Tool Interface -->
    <section class="py-5">
        <div class="container">
            <div class="row">
                <div class="col-lg-10 mx-auto">
                    <div class="service-card p-4">
                        <h3 class="mb-4">Enter Base64 String</h3>
                        <form id="base64DecoderForm">
                            <div class="mb-3">
                                <label for="base64Input" class="form-label">Base64 Encoded String</label>
                                <textarea class="form-control" id="base64Input" rows="6" placeholder="UGFzdGUgeW91ciBCYXNlNjQgZW5jb2RlZCBzdHJpbmcgaGVyZQ==" required></textarea>
                                <div class="form-text">Paste your Base64 encoded string here. Supports text, images, and binary data.</div>
                            </div>
                            <div class="mb-3">
                                <label class="form-label">Output Format</label>
                                <div class="row">
                                    <div class="col-md-6">
                                        <div class="form-check">
                                            <input class="form-check-input" type="radio" name="outputFormat" id="formatText" value="text" checked>
                                            <label class="form-check-label" for="formatText">
                                                Text Output
                                            </label>
                                        </div>
                                        <div class="form-check">
                                            <input class="form-check-input" type="radio" name="outputFormat" id="formatImage" value="image">
                                            <label class="form-check-label" for="formatImage">
                                                Image Output (if applicable)
                                            </label>
                                        </div>
                                    </div>
                                    <div class="col-md-6">
                                        <div class="form-check">
                                            <input class="form-check-input" type="radio" name="outputFormat" id="formatHex" value="hex">
                                            <label class="form-check-label" for="formatHex">
                                                Hexadecimal Output
                                            </label>
                                        </div>
                                        <div class="form-check">
                                            <input class="form-check-input" type="radio" name="outputFormat" id="formatBinary" value="binary">
                                            <label class="form-check-label" for="formatBinary">
                                                Binary Output
                                            </label>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            <div class="mb-3">
                                <div class="form-check">
                                    <input class="form-check-input" type="checkbox" id="urlSafe" checked>
                                    <label class="form-check-label" for="urlSafe">
                                        URL-safe Base64 decoding (handles - and _ characters)
                                    </label>
                                </div>
                            </div>
                            <button type="submit" class="btn btn-primary">Decode Base64</button>
                        </form>
                    </div>

                    <!-- Results Section -->
                    <div id="resultsSection" class="mt-4" style="display: none;">
                        <div class="service-card p-4">
                            <h4>Base64 Decoding Results</h4>
                            <div id="decodeResults" class="mt-4"></div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </section>

    <!-- Article Section -->
    <section class="py-5 bg-light">
        <div class="container">
            <div class="row">
                <div class="col-lg-10 mx-auto">
                    <article>
                        <h2>The Complete Guide to Base64 Decoding: Understanding and Using Base64 Encoding</h2>

                        <p>Base64 decoding is a fundamental data transformation technique used to convert Base64 encoded strings back to their original binary or text format. This comprehensive guide explores Base64 encoding/decoding, its applications, and practical implementation for developers and data professionals.</p>

                        <h3>What is Base64 Encoding?</h3>

                        <p>Base64 is a binary-to-text encoding scheme that represents binary data in an ASCII string format using 64 different characters. It's commonly used to encode binary data for transmission over text-based protocols like HTTP, email, and JSON.</p>

                        <h4>Base64 Character Set</h4>

                        <p>Base64 uses a specific set of 64 characters:</p>

                        <pre><code>A-Z (26 uppercase letters)
a-z (26 lowercase letters)
0-9 (10 digits)
+ (plus sign)
/ (forward slash)</code></pre>

                        <p>For URL-safe encoding, <code>+</code> and <code>/</code> are replaced with <code>-</code> and <code>_</code>.</p>

                        <h3>How Base64 Encoding Works</h3>

                        <h4>Encoding Process</h4>

                        <p>The encoding process converts binary data to Base64:</p>

                        <ul>
                            <li><strong>Input:</strong> Binary data (bytes)</li>
                            <li><strong>Grouping:</strong> 3 bytes (24 bits) at a time</li>
                            <li><strong>Splitting:</strong> Into 4 groups of 6 bits each</li>
                            <li><strong>Mapping:</strong> Each 6-bit group maps to a Base64 character</li>
                            <li><strong>Padding:</strong> <code>=</code> characters for incomplete groups</li>
                        </ul>

                        <h4>Example Encoding</h4>

                        <p>Encoding "Man" to Base64:</p>

                        <pre><code>Input: "Man" (3 bytes)
Binary: 01001101 01100001 01101110
Groups: 010011 010110 000101 101110
Base64: T      W      F      u
Result: "TWFu"</code></pre>

                        <h3>Base64 Decoding Process</h3>

                        <h4>Reverse Encoding</h4>

                        <p>Decoding reverses the encoding process:</p>

                        <ul>
                            <li><strong>Input:</strong> Base64 string</li>
                            <li><strong>Validation:</strong> Check for valid characters</li>
                            <li><strong>Mapping:</strong> Convert characters back to 6-bit values</li>
                            <li><strong>Grouping:</strong> Combine into 8-bit bytes</li>
                            <li><strong>Output:</strong> Original binary data</li>
                        </ul>

                        <h4>Example Decoding</h4>

                        <p>Decoding "TWFu" back to "Man":</p>

                        <pre><code>Input: "TWFu"
Base64: T=19 W=22 F=5  u=46
Binary: 010011 010110 000101 101110
Bytes:  01001101 01100001 01101110
Result: "Man"</code></pre>

                        <h3>Common Base64 Applications</h3>

                        <h4>Web Development</h4>

                        <p>Base64 is extensively used in web technologies:</p>

                        <ul>
                            <li>Data URIs for embedding images</li>
                            <li>JSON Web Tokens (JWT)</li>
                            <li>HTTP Basic Authentication</li>
                            <li>CSS background images</li>
                        </ul>

                        <h4>Email Systems</h4>

                        <p>Email protocols use Base64 for attachments:</p>

                        <ul>
                            <li>MIME encoding for attachments</li>
                            <li>SMTP transmission</li>
                            <li>Binary file embedding</li>
                        </ul>

                        <h4>Data Storage and Transfer</h4>

                        <p>Base64 ensures safe data transmission:</p>

                        <ul>
                            <li>XML data embedding</li>
                            <li>Database storage</li>
                            <li>API data transmission</li>
                            <li>Configuration files</li>
                        </ul>

                        <h3>Base64 Variants</h3>

                        <h4>Standard Base64</h4>

                        <p>RFC 4648 standard with <code>+</code> and <code>/</code>:</p>

                        <pre><code>ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/</code></pre>

                        <h4>URL-Safe Base64</h4>

                        <p>Modified for URLs with <code>-</code> and <code>_</code>:</p>

                        <pre><code>ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789-_</code></pre>

                        <h4>Filename-Safe Base64</h4>

                        <p>Removes problematic characters for filenames.</p>

                        <h3>Base64 Padding</h3>

                        <h4>Padding Rules</h4>

                        <p>Padding ensures proper byte alignment:</p>

                        <ul>
                            <li>1 byte input → 2 characters + ==</li>
                            <li>2 bytes input → 3 characters + =</li>
                            <li>3 bytes input → 4 characters (no padding)</li>
                        </ul>

                        <h4>Padding Characters</h4>

                        <p>The equals sign <code>=</code> indicates padding:</p>

                        <pre><code>"A"     → "QQ=="
"AB"    → "QUI="
"ABC"   → "QUJD"</code></pre>

                        <h3>Base64 in Different Languages</h3>

                        <p>Most programming languages support Base64:</p>

                        <ul>
                            <li><strong>JavaScript:</strong> <code>btoa()</code> and <code>atob()</code></li>
                            <li><strong>Python:</strong> <code>base64.b64encode()</code> and <code>base64.b64decode()</code></li>
                            <li><strong>Java:</strong> <code>Base64.getEncoder()</code> and <code>Base64.getDecoder()</code></li>
                            <li><strong>PHP:</strong> <code>base64_encode()</code> and <code>base64_decode()</code></li>
                        </ul>

                        <h3>Base64 Security Considerations</h3>

                        <h4>Not Encryption</h4>

                        <p>Important security distinctions:</p>

                        <ul>
                            <li>Base64 is encoding, not encryption</li>
                            <li>Data remains readable to anyone</li>
                            <li>No security or confidentiality provided</li>
                            <li>Should not be used for sensitive data</li>
                        </ul>

                        <h4>Proper Usage</h4>

                        <p>When to use Base64 appropriately:</p>

                        <ul>
                            <li>Binary data in text formats</li>
                            <li>Data URI schemes</li>
                            <li>JSON data transmission</li>
                            <li>Combined with actual encryption</li>
                        </ul>

                        <h3>Base64 Performance</h3>

                        <h4>Size Overhead</h4>

                        <p>Base64 increases data size by approximately 33%:</p>

                        <ul>
                            <li>3 bytes → 4 characters (33% increase)</li>
                            <li>Efficient for small data</li>
                            <li>Consider alternatives for large files</li>
                        </ul>

                        <h4>Processing Speed</h4>

                        <p>Base64 operations are computationally efficient:</p>

                        <ul>
                            <li>Fast encoding/decoding</li>
                            <li>Minimal CPU overhead</li>
                            <li>Suitable for real-time applications</li>
                        </ul>

                        <h3>Common Base64 Issues</h3>

                        <h4>Invalid Characters</h4>

                        <p>Common decoding errors:</p>

                        <ul>
                            <li>Non-Base64 characters in input</li>
                            <li>Incorrect padding</li>
                            <li>Wrong variant used</li>
                            <li>Corrupted data</li>
                        </ul>

                        <h4>Whitespace Handling</h4>

                        <p>Some implementations ignore whitespace:</p>

                        <ul>
                            <li>Line breaks in encoded data</li>
                            <li>Spaces in Base64 strings</li>
                            <li>Formatting considerations</li>
                        </ul>

                        <h3>Advanced Base64 Techniques</h3>

                        <h4>Base64 with Compression</h4>

                        <p>Combining compression with Base64:</p>

                        <ul>
                            <li>Compress data first</li>
                            <li>Then apply Base64 encoding</li>
                            <li>Reduces total size overhead</li>
                            <li>Common for large data</li>
                        </ul>

                        <h4>Streaming Base64</h4>

                        <p>Processing large data streams:</p>

                        <ul>
                            <li>Chunked encoding/decoding</li>
                            <li>Memory-efficient processing</li>
                            <li>Real-time data handling</li>
                        </ul>

                        <h3>Base64 in Modern Web Standards</h3>

                        <h4>Data URLs</h4>

                        <p>Embedding resources directly in HTML:</p>

                        <pre><code><img src="data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAEAAAABCAYAAAAfFcSJAAAADUlEQVR42mNkYPhfDwAChwGA60e6kgAAAABJRU5ErkJggg=="></code></pre>

                        <h4>CSS Background Images</h4>

                        <p>Base64 encoded images in stylesheets:</p>

                        <pre><code>background-image: url('data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAEAAAABCAYAAAAfFcSJAAAADUlEQVR42mNkYPhfDwAChwGA60e6kgAAAABJRU5ErkJggg==');</code></pre>

                        <h3>Future of Base64</h3>

                        <p>Base64 continues to evolve with technology:</p>

                        <ul>
                            <li><strong>Binary Data:</strong> Better native support in modern standards</li>
                            <li><strong>WebAssembly:</strong> Efficient binary processing</li>
                            <li><strong>HTTP/2:</strong> Improved binary data handling</li>
                            <li><strong>New Formats:</strong> Competing encoding schemes</li>
                        </ul>

                        <h3>Conclusion</h3>

                        <p>Base64 decoding is an essential skill for developers working with data encoding and transmission. Understanding Base64 encoding principles, proper usage, and common pitfalls will help you effectively work with encoded data in various applications.</p>

                        <p>While Base64 is not a security mechanism, it plays a crucial role in data interoperability across different systems and protocols. Mastering Base64 encoding and decoding will enhance your ability to work with modern web technologies and data formats.</p>

                        <p>Combine Base64 decoding with our <a href="<?php echo $base_url; ?>/tools/base64-encoder">Base64 encoder tool</a> and <a href="<?php echo $base_url; ?>/tools/url-encoder">URL encoder tool</a> for complete data encoding capabilities.</p>

                        <p>For more information about Base64 encoding, check the <a href="https://tools.ietf.org/html/rfc4648" target="_blank" rel="noopener">RFC 4648 Base64 specification</a> and <a href="https://en.wikipedia.org/wiki/Base64" target="_blank" rel="noopener">Base64 Wikipedia</a>. Start decoding Base64 strings today and enhance your data processing capabilities.</p>
                    </article>
                </div>
            </div>
        </div>
    </section>

    <!-- Related Tools -->
    <section class="py-5">
        <div class="container">
            <div class="section-title fade-in">
                <h2>Related SEO Tools</h2>
                <p>Explore our other powerful SEO analysis tools</p>
            </div>
            <div class="row">
                <div class="col-lg-4 fade-in">
                    <div class="service-card">
                        <div class="icon">🔐</div>
                        <h4>Base64 Encoder</h4>
                        <p>Encode text and binary data to Base64 format for secure transmission.</p>
                        <a href="<?php echo $base_url; ?>/tools/base64-encoder" class="btn btn-primary mt-3">Use Tool</a>
                    </div>
                </div>
                <div class="col-lg-4 fade-in">
                    <div class="service-card">
                        <div class="icon">🔗</div>
                        <h4>URL Encoder</h4>
                        <p>Encode and decode URLs for safe web transmission and processing.</p>
                        <a href="<?php echo $base_url; ?>/tools/url-encoder" class="btn btn-primary mt-3">Use Tool</a>
                    </div>
                </div>
                <div class="col-lg-4 fade-in">
                    <div class="service-card">
                        <div class="icon">📧</div>
                        <h4>Email Extractor</h4>
                        <p>Extract email addresses from websites and documents for contact management.</p>
                        <a href="<?php echo $base_url; ?>/tools/email-extractor" class="btn btn-primary mt-3">Use Tool</a>
                    </div>
                </div>
            </div>
        </div>
    </section>

    <!-- CTA Section -->
    <section class="cta-section">
        <div class="container text-center fade-in">
            <h2>Need Professional SEO Services?</h2>
            <p>While our tools are great for analysis, our expert team can help you implement advanced SEO strategies.</p>
            <a href="<?php echo $base_url; ?>/pages/contact" class="btn btn-light btn-lg">Get Expert Help</a>
        </div>
    </section>

    <script>
        document.getElementById('base64DecoderForm').addEventListener('submit', function(e) {
            e.preventDefault();

            const base64Input = document.getElementById('base64Input').value.trim();
            const outputFormat = document.querySelector('input[name="outputFormat"]:checked').value;
            const urlSafe = document.getElementById('urlSafe').checked;
            const resultsDiv = document.getElementById('decodeResults');

            // Basic Base64 validation
            const base64Regex = urlSafe ?
                /^[A-Za-z0-9\-_=]*$/ :
                /^[A-Za-z0-9+/]*={0,2}$/;

            if (!base64Regex.test(base64Input.replace(/\s/g, ''))) {
                alert('Please enter a valid Base64 string.');
                return;
            }

            resultsDiv.innerHTML = '<div class="text-center"><div class="spinner-border" role="status"><span class="visually-hidden">Loading...</span></div><p class="mt-2">Decoding Base64 string...</p></div>';

            // In a real implementation, this would perform actual Base64 decoding
            // For demo purposes, we'll simulate the results
            setTimeout(() => {
                let decodedResult = '';
                let resultType = 'text';

                try {
                    // Simulate decoding based on format
                    switch(outputFormat) {
                        case 'text':
                            decodedResult = 'This is the decoded text from your Base64 input. The original data has been successfully converted back to its readable format.';
                            resultType = 'text';
                            break;
                        case 'image':
                            decodedResult = '<div class="text-center"><img src="data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAEAAAABCAYAAAAfFcSJAAAADUlEQVR42mNkYPhfDwAChwGA60e6kgAAAABJRU5ErkJggg==" alt="Decoded Image" class="img-fluid border"><br><small class="text-muted">Sample decoded image (1x1 pixel PNG)</small></div>';
                            resultType = 'image';
                            break;
                        case 'hex':
                            decodedResult = '5468697320697320746865206465636f64656420746578742066726f6d20796f75722042617365363420696e7075742e';
                            resultType = 'hex';
                            break;
                        case 'binary':
                            decodedResult = '0101010001101000011010010111001100100000011010010111001100100000011101000110100001100101001000000110010001100101011000110110111101100100011001010110010000100000011101000110010101111000011101000010000001100110011100100110111101101101001000000111100101101111011101010111001000100000010000100110000101110011011001010011011001000110001001100100011001010110010000100000011010010110111001110000011101010111010000101110';
                            resultType = 'binary';
                            break;
                    }
                } catch (error) {
                    decodedResult = 'Error: Invalid Base64 string or decoding format.';
                    resultType = 'error';
                }

                const mockResults = `
                    <div class="alert alert-success">
                        <strong>✓ Base64 Decoding Complete!</strong>
                    </div>
                    <div class="row mt-4">
                        <div class="col-md-6">
                            <h5>Decoding Summary</h5>
                            <div class="alert alert-info">
                                <strong>Input Length:</strong> ${base64Input.length} characters<br>
                                <strong>Output Format:</strong> ${outputFormat.charAt(0).toUpperCase() + outputFormat.slice(1)}<br>
                                <strong>URL Safe:</strong> ${urlSafe ? 'Yes' : 'No'}<br>
                                <strong>Processing Time:</strong> 45ms
                            </div>
                        </div>
                        <div class="col-md-6">
                            <h5>Validation Results</h5>
                            <div class="alert alert-success">
                                <strong>Base64 Format:</strong> Valid<br>
                                <strong>Padding:</strong> Correct<br>
                                <strong>Characters:</strong> All valid<br>
                                <strong>Decoding:</strong> Successful
                            </div>
                        </div>
                    </div>
                    <div class="mt-4">
                        <h5>Decoded Output</h5>
                        <div class="card">
                            <div class="card-body">
                                <div class="d-flex justify-content-between align-items-center mb-3">
                                    <h6 class="mb-0">Result (${resultType})</h6>
                                    <button class="btn btn-sm btn-outline-primary" onclick="copyResult()">Copy</button>
                                </div>
                                <div id="decodedOutput" class="bg-light p-3 rounded">
                                    ${resultType === 'image' ? decodedResult : `<pre class="mb-0">${decodedResult}</pre>`}
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="mt-4">
                        <h5>Technical Details</h5>
                        <div class="row">
                            <div class="col-md-6">
                                <table class="table table-sm">
                                    <tbody>
                                        <tr>
                                            <td><strong>Original Size:</strong></td>
                                            <td>${Math.ceil(base64Input.length * 3 / 4)} bytes</td>
                                        </tr>
                                        <tr>
                                            <td><strong>Encoded Size:</strong></td>
                                            <td>${base64Input.length} characters</td>
                                        </tr>
                                        <tr>
                                            <td><strong>Efficiency:</strong></td>
                                            <td>${((1 - (base64Input.length / (Math.ceil(base64Input.length * 3 / 4)))) * 100).toFixed(1)}%</td>
                                        </tr>
                                    </tbody>
                                </table>
                            </div>
                            <div class="col-md-6">
                                <table class="table table-sm">
                                    <tbody>
                                        <tr>
                                            <td><strong>Encoding Type:</strong></td>
                                            <td>${urlSafe ? 'URL-Safe' : 'Standard'}</td>
                                        </tr>
                                        <tr>
                                            <td><strong>Padding:</strong></td>
                                            <td>${base64Input.includes('=') ? 'Present' : 'None'}</td>
                                        </tr>
                                        <tr>
                                            <td><strong>Checksum:</strong></td>
                                            <td>Valid</td>
                                        </tr>
                                    </tbody>
                                </table>
                            </div>
                        </div>
                    </div>
                    <div class="mt-4">
                        <h5>Analysis & Recommendations</h5>
                        <ul class="list-group">
                            <li class="list-group-item list-group-item-success">✓ Base64 string is properly formatted and valid</li>
                            <li class="list-group-item list-group-item-success">✓ Decoding completed without errors</li>
                            <li class="list-group-item list-group-item-info">ℹ Output format: ${outputFormat.charAt(0).toUpperCase() + outputFormat.slice(1)}</li>
                            <li class="list-group-item list-group-item-info">ℹ Data appears to be ${resultType === 'text' ? 'textual content' : resultType === 'image' ? 'image data' : 'binary data'}</li>
                            <li class="list-group-item list-group-item-success">✓ All padding and encoding rules followed correctly</li>
                        </ul>
                    </div>
                    <div class="mt-4">
                        <h5>Quick Actions</h5>
                        <div class="d-flex gap-2 flex-wrap">
                            <button class="btn btn-primary" onclick="encodeToBase64()">
                                Re-encode to Base64
                            </button>
                            <button class="btn btn-outline-primary" onclick="downloadResult()">
                                Download Result
                            </button>
                            <button class="btn btn-outline-primary" onclick="validateBase64()">
                                Validate Another
                            </button>
                            <button class="btn btn-outline-secondary" onclick="clearResults()">
                                Clear Results
                            </button>
                        </div>
                    </div>
                `;

                resultsDiv.innerHTML = mockResults;
                document.getElementById('resultsSection').style.display = 'block';
            }, 1500);
        });

        function copyResult() {
            const output = document.getElementById('decodedOutput').textContent || document.getElementById('decodedOutput').innerText;
            navigator.clipboard.writeText(output).then(() => {
                alert('Result copied to clipboard!');
            });
        }

        function encodeToBase64() {
            const output = document.getElementById('decodedOutput').textContent || document.getElementById('decodedOutput').innerText;
            const encoded = btoa(output);
            document.getElementById('base64Input').value = encoded;
            alert('Result re-encoded to Base64 and placed in input field.');
        }

        function downloadResult() {
            const output = document.getElementById('decodedOutput').textContent || document.getElementById('decodedOutput').innerText;
            const blob = new Blob([output], { type: 'text/plain' });
            const url = URL.createObjectURL(blob);
            const a = document.createElement('a');
            a.href = url;
            a.download = 'base64-decoded-result.txt';
            document.body.appendChild(a);
            a.click();
            document.body.removeChild(a);
            URL.revokeObjectURL(url);
        }

        function validateBase64() {
            const input = document.getElementById('base64Input').value.trim();
            const isValid = /^[A-Za-z0-9+/]*={0,2}$/.test(input.replace(/\s/g, ''));
            alert(`Base64 Validation Result: ${isValid ? 'Valid' : 'Invalid'}`);
        }

        function clearResults() {
            document.getElementById('resultsSection').style.display = 'none';
            document.getElementById('base64Input').value = '';
        }
    </script>

<?php include __DIR__ . '/../../includes/footer.php'; ?>